"""Test OracleDB driver implementation - Synchronous Tests."""

from __future__ import annotations

from typing import Any, Literal

import pyarrow as pa
import pytest
from pytest_databases.docker.oracle import OracleService

from sqlspec.adapters.oracledb import OracleSyncConfig, OracleSyncPoolConfig

ParamStyle = Literal["positional_binds", "dict_binds"]

# --- Sync Fixtures ---


@pytest.fixture
def oracle_sync_session(oracle_23ai_service: OracleService) -> OracleSyncConfig:
    """Create an Oracle synchronous session."""
    return OracleSyncConfig(
        pool_config=OracleSyncPoolConfig(
            host=oracle_23ai_service.host,
            port=oracle_23ai_service.port,
            service_name=oracle_23ai_service.service_name,
            user=oracle_23ai_service.user,
            password=oracle_23ai_service.password,
        )
    )


# --- Sync Tests ---


@pytest.mark.parametrize(
    ("params", "style"),
    [
        pytest.param(("test_name",), "positional_binds", id="positional_binds"),
        pytest.param({"name": "test_name"}, "dict_binds", id="dict_binds"),
    ],
)
@pytest.mark.skip(
    reason="Oracle does not support RETURNING multiple columns directly in the required syntax for this method."
)
@pytest.mark.xdist_group("oracle")
def test_sync_insert_returning(oracle_sync_session: OracleSyncConfig, params: Any, style: ParamStyle) -> None:
    """Test synchronous insert returning functionality with Oracle parameter styles."""
    with oracle_sync_session.provide_session() as driver:
        sql = """
        CREATE TABLE test_table (
            id NUMBER GENERATED BY DEFAULT ON NULL AS IDENTITY PRIMARY KEY,
            name VARCHAR2(50)
        )
        """
        driver.execute_script(sql)

        if style == "positional_binds":
            sql = "INSERT INTO test_table (name) VALUES (:1) RETURNING id, name"
            exec_params = params
        else:  # dict_binds
            # Workaround: Use positional binds due to DPY-4009
            sql = "INSERT INTO test_table (name) VALUES (:1) RETURNING id, name"
            exec_params = (params["name"],)

        result = driver.insert_update_delete_returning(sql, exec_params)
        assert result is not None
        # Oracle often returns column names in uppercase
        assert result["NAME"] == "test_name"
        assert result["ID"] is not None
        assert isinstance(result["ID"], int)
        driver.execute_script(
            "BEGIN EXECUTE IMMEDIATE 'DROP TABLE test_table'; EXCEPTION WHEN OTHERS THEN IF SQLCODE != -942 THEN RAISE; END IF; END;"
        )


@pytest.mark.parametrize(
    ("params", "style"),
    [
        pytest.param(("test_name",), "positional_binds", id="positional_binds"),
        pytest.param({"name": "test_name"}, "dict_binds", id="dict_binds"),
    ],
)
@pytest.mark.xdist_group("oracle")
def test_sync_select(oracle_sync_session: OracleSyncConfig, params: Any, style: ParamStyle) -> None:
    """Test synchronous select functionality with Oracle parameter styles."""
    with oracle_sync_session.provide_session() as driver:
        sql = """
        CREATE TABLE test_table (
            id NUMBER GENERATED BY DEFAULT ON NULL AS IDENTITY PRIMARY KEY,
            name VARCHAR2(50)
        )
        """
        driver.execute_script(sql)

        if style == "positional_binds":
            insert_sql = "INSERT INTO test_table (name) VALUES (:1)"
            select_sql = "SELECT name FROM test_table WHERE name = :1"
            insert_params = params
            select_params = params
        else:  # dict_binds
            # Workaround: Use positional binds due to DPY-4009
            insert_sql = "INSERT INTO test_table (name) VALUES (:1)"
            select_sql = "SELECT name FROM test_table WHERE name = :1"
            insert_params = (params["name"],)
            select_params = (params["name"],)

        driver.insert_update_delete(insert_sql, insert_params)

        results = driver.select(select_sql, select_params)
        assert len(results) == 1
        assert results[0]["NAME"] == "test_name"
        driver.execute_script(
            "BEGIN EXECUTE IMMEDIATE 'DROP TABLE test_table'; EXCEPTION WHEN OTHERS THEN IF SQLCODE != -942 THEN RAISE; END IF; END;"
        )


@pytest.mark.parametrize(
    ("params", "style"),  # Keep parametrization for structure, even if params unused for select_value
    [
        pytest.param(("test_name",), "positional_binds", id="positional_binds"),
        pytest.param({"name": "test_name"}, "dict_binds", id="dict_binds"),
    ],
)
@pytest.mark.xdist_group("oracle")
def test_sync_select_value(oracle_sync_session: OracleSyncConfig, params: Any, style: ParamStyle) -> None:
    """Test synchronous select_value functionality with Oracle parameter styles."""
    with oracle_sync_session.provide_session() as driver:
        sql = """
        CREATE TABLE test_table (
            id NUMBER GENERATED BY DEFAULT ON NULL AS IDENTITY PRIMARY KEY,
            name VARCHAR2(50)
        )
        """
        driver.execute_script(sql)

        # Workaround: Use positional binds for setup insert due to DPY-4009 error with dict_binds
        if style == "positional_binds":
            setup_value = params[0]
        else:  # dict_binds
            setup_value = params["name"]
        setup_params_tuple = (setup_value,)
        insert_sql_setup = "INSERT INTO test_table (name) VALUES (:1)"
        driver.insert_update_delete(insert_sql_setup, setup_params_tuple)

        # Select a literal value using Oracle's DUAL table
        select_sql = "SELECT 'test_value' FROM dual"
        value = driver.select_value(select_sql)
        assert value == "test_value"
        driver.execute_script(
            "BEGIN EXECUTE IMMEDIATE 'DROP TABLE test_table'; EXCEPTION WHEN OTHERS THEN IF SQLCODE != -942 THEN RAISE; END IF; END;"
        )


@pytest.mark.xdist_group("oracle")
def test_sync_select_arrow(oracle_sync_session: OracleSyncConfig) -> None:
    """Test synchronous select_arrow functionality."""
    with oracle_sync_session.provide_session() as driver:
        sql = """
        CREATE TABLE test_table (
            id NUMBER GENERATED BY DEFAULT ON NULL AS IDENTITY PRIMARY KEY,
            name VARCHAR2(50)
        )
        """
        driver.execute_script(sql)

        # Insert test record using positional binds
        insert_sql = "INSERT INTO test_table (name) VALUES (:1)"
        driver.insert_update_delete(insert_sql, ("arrow_name",))

        # Select and verify with select_arrow using positional binds
        select_sql = "SELECT name, id FROM test_table WHERE name = :1"
        arrow_table = driver.select_arrow(select_sql, ("arrow_name",))

        assert isinstance(arrow_table, pa.Table)
        assert arrow_table.num_rows == 1
        assert arrow_table.num_columns == 2
        # Oracle returns uppercase column names by default
        assert arrow_table.column_names == ["NAME", "ID"]
        assert arrow_table.column("NAME").to_pylist() == ["arrow_name"]
        # Check ID exists and is a number (exact value depends on IDENTITY)
        assert arrow_table.column("ID").to_pylist()[0] is not None
        assert isinstance(arrow_table.column("ID").to_pylist()[0], (int, float))  # Oracle NUMBER maps to float/Decimal
        driver.execute_script(
            "BEGIN EXECUTE IMMEDIATE 'DROP TABLE test_table'; EXCEPTION WHEN OTHERS THEN IF SQLCODE != -942 THEN RAISE; END IF; END;"
        )
