"""
Test script for Interactive SQL Shell

Validates that shell components work correctly without
running the full TUI (to avoid terminal keybinding conflicts).
"""

import sys
import tempfile
from pathlib import Path

# Test imports
print("Testing imports...")
try:
    from sqlstream.cli.shell import SQLShellApp, QueryEditor, ResultsViewer, StatusBar
    
    # Verify they accept kwargs (like id)
    rv = ResultsViewer(id="test-viewer")
    sb = StatusBar(id="test-status")
    print("✅ All shell components imported and instantiated successfully")
except ImportError as e:
    print(f"❌ Import error: {e}")
    sys.exit(1)

# Test query execution engine
print("\nTesting query execution...")
try:
    from sqlstream.core.query import QueryInline
    
    # Create test data
    test_csv = tempfile.NamedTemporaryFile(mode='w', suffix='.csv', delete=False)
    test_csv.write("name,age,city,salary\n")
    test_csv.write("Alice,30,NYC,90000\n")
    test_csv.write("Bob,25,LA,75000\n")
    test_csv.write("Charlie,35,SF,85000\n")
    test_csv.close()
    
    # Test query
    q = QueryInline()
    result = q.sql(f"SELECT * FROM '{test_csv.name}' WHERE age > 25")
    results = result.to_list()
    
    # Test SQLSuggester
    print("\nTesting Auto-completion...")
    from sqlstream.cli.shell import SQLSuggester
    import asyncio
    
    suggester = SQLSuggester()
    
    # Test keyword suggestion
    suggestion = asyncio.run(suggester.get_suggestion("SEL"))
    assert suggestion == "SELECT"
    print("✅ Auto-completion: 'SEL' -> 'SELECT'")
    
    # Test table suggestion
    suggester.update_schema({"employees.csv": {"name": "string"}})
    suggestion = asyncio.run(suggester.get_suggestion("empl"))
    assert suggestion == "employees.csv"
    print("✅ Auto-completion: 'empl' -> 'employees.csv'")
    
    # Test column suggestion
    suggestion = asyncio.run(suggester.get_suggestion("na"))
    assert suggestion == "name"
    print("✅ Auto-completion: 'na' -> 'name'")
    
    print(f"✅ Query executed: {len(results)} rows returned")
    print(f"   Sample row: {results[0]}")
    
    # Cleanup
    Path(test_csv.name).unlink()
    
except Exception as e:
    print(f"❌ Query execution error: {e}")
    import traceback
    traceback.print_exc()
    sys.exit(1)

# Test shell app initialization
print("\nTesting shell app initialization...")
try:
    app = SQLShellApp(initial_file=None, history_file="/tmp/test_history")
    print("✅ Shell app initialized successfully")
    print(f"   Title: {app.title}")
    print(f"   History file: {app.history_file}")
except Exception as e:
    print(f"❌ Shell initialization error: {e}")
    import traceback
    traceback.print_exc()
    sys.exit(1)

    # Test Pagination
    print("\nTesting Pagination...")
    app.last_results = [{"id": i, "value": f"row{i}"} for i in range(250)]
    app.filtered_results = app.last_results.copy()
    app.page_size = 100
    app.current_page = 0
    
    assert len(app.filtered_results) == 250
    total_pages = (250 + 100 - 1) // 100
    assert total_pages == 3
    print(f"✅ Pagination: 250 rows divided into {total_pages} pages")
    
    # Test page navigation logic
    if app.current_page < total_pages - 1:
        app.current_page += 1
    assert app.current_page == 1
    print("✅ Pagination: Next page logic works")
    
    if app.current_page > 0:
        app.current_page -= 1
    assert app.current_page == 0
    print("✅ Pagination: Prev page logic works")
    
    # Test Filtering
    print("\nTesting Filtering...")
    app.last_results = [
        {"name": "Alice", "city": "NYC"},
        {"name": "Bob", "city": "LA"},
        {"name": "Charlie", "city": "NYC"}
    ]
    app.filter_text = "nyc"
    filtered = app._apply_filter(app.last_results)
    assert len(filtered) == 2
    assert all("nyc" in str(r).lower() for r in filtered)
    print("✅ Filtering: Filtered 3 rows to 2 (case-insensitive)")
    
    # Test Sorting
    print("\nTesting Sorting...")
    app.last_results = [
        {"id": 3, "name": "Charlie"},
        {"id": 1, "name": "Alice"},
        {"id": 2, "name": "Bob"}
    ]
    app.sort_column = "id"
    app.sort_reverse = False
    sorted_results = app._apply_sort(app.last_results)
    assert sorted_results[0]["id"] == 1
    assert sorted_results[2]["id"] == 3
    print("✅ Sorting: Sorted by 'id' ascending")
    
    # Test reverse sort
    app.sort_reverse = True
    sorted_results = app._apply_sort(app.last_results)
    assert sorted_results[0]["id"] == 3
    assert sorted_results[2]["id"] == 1
    print("✅ Sorting: Sorted by 'id' descending")

# Test CLI command registration
print("\nTesting CLI command registration...")
try:
    from sqlstream.cli.main import cli
    
    # Check if shell command exists
    commands = [cmd.name for cmd in cli.commands.values()]
    if 'shell' in commands:
        print("✅ 'shell' command registered in CLI")
        print(f"   Available commands: {', '.join(commands)}")
    else:
        print("❌ 'shell' command not found in CLI")
        sys.exit(1)
        
except Exception as e:
    print(f"❌ CLI test error: {e}")
    import traceback
    traceback.print_exc()
    sys.exit(1)

# Summary
print("\n" + "="*60)
print("✅ All component tests passed!")
print("="*60)
print("\nThe interactive shell is ready to use!")
print("\nTo test manually (in a real terminal, not VSCode):")
print("  python -m sqlstream shell /tmp/test_shell.csv")
print("\nKeybindings:")
print("  Ctrl+Enter or Ctrl+E  - Execute query")
print("  Ctrl+L                - Clear editor")
print("  Ctrl+D                - Exit")
print("  F1                    - Help")
print("\n⚠️  Note: VSCode terminal has Ctrl+J conflicts with Textual.")
print("   For best experience, use a native terminal (gnome-terminal, iTerm2, etc.)")
