use squawk_syntax::{
    Parse, SourceFile,
    ast::{self, AstNode},
};

use crate::{Linter, Rule, Violation};

pub(crate) fn ban_alter_domain_with_add_constraint(ctx: &mut Linter, parse: &Parse<SourceFile>) {
    let file = parse.tree();
    for stmt in file.stmts() {
        if let ast::Stmt::AlterDomain(alter_domain) = stmt {
            if let Some(ast::AlterDomainAction::AddConstraint(add_constraint)) =
                alter_domain.action()
            {
                ctx.report(Violation::for_node(
                    Rule::BanAlterDomainWithAddConstraint,
                        "Domains with constraints have poor support for online migrations. Use table and column constraints instead.".into(),
                        add_constraint.syntax(),
                    ))
            }
        }
    }
}

#[cfg(test)]
mod test {
    use insta::assert_snapshot;

    use crate::Rule;
    use crate::test_utils::{lint_errors, lint_ok};

    #[test]
    fn err() {
        let sql = r#"
     ALTER DOMAIN domain_name ADD CONSTRAINT constraint_name CHECK (value > 0);
        "#;
        assert_snapshot!(lint_errors(sql, Rule::BanAlterDomainWithAddConstraint));
    }

    #[test]
    fn ok() {
        let sql = r#"
     ALTER DOMAIN domain_name_2 SET NOT NULL;
     ALTER DOMAIN domain_name_3 DROP CONSTRAINT other_domain_name;
     ALTER DOMAIN domain_name_4 RENAME CONSTRAINT constraint_name TO other_constraint_name;
     ALTER DOMAIN domain_name_5 RENAME TO other_domain_name;
     ALTER DOMAIN domain_name_6 VALIDATE CONSTRAINT constraint_name;
     ALTER DOMAIN domain_name_7 OWNER TO you;
     ALTER DOMAIN domain_name_8 SET SCHEMA foo;
        "#;
        lint_ok(sql, Rule::BanAlterDomainWithAddConstraint);
    }
}
