from stackifyapm.instrumentation.packages.base import AbstractInstrumentedModule
from stackifyapm.traces import CaptureSpan
from stackifyapm.utils import compat, get_method_name


class PythonMemcachedInstrumentation(AbstractInstrumentedModule):
    name = "python_memcached"

    memcached_method_list = [
        "add",
        "append",
        "cas",
        "decr",
        "delete",
        "delete_multi",
        "disconnect_all",
        "flush_all",
        "get",
        "get_multi",
        "get_slabs",
        "get_stats",
        "gets",
        "incr",
        "prepend",
        "replace",
        "set",
        "set_multi",
        "touch",
    ]

    def get_instrument_list(self):
        method_list = [("memcache", "Client.{}".format(method)) for method in self.memcached_method_list]
        method_list += [("pymemcache.client.base", "Client.{}".format(method)) for method in self.memcached_method_list]
        return method_list

    def call(self, module, method, wrapped, instance, args, kwargs):
        name = self.get_wrapped_name(wrapped, instance, method)
        method_name = get_method_name(method)
        extra_data = {
            "wrapped_method": "execute",
            "provider": self.name,
            "type": "Cache",
            "sub_type": "cache",
            "operation": method_name,
        }
        cache_name = args and args[0] or None
        if cache_name:
            if isinstance(cache_name, compat.string_types):
                extra_data['cache_name'] = cache_name
                extra_data['cache_key'] = cache_name.split(':')[-1]
            elif isinstance(cache_name, compat.list_type):
                extra_data['cache_name'] = cache_name
                extra_data['cache_key'] = [n.split(':')[-1] for n in cache_name]
            elif isinstance(cache_name, compat.dict_type):
                extra_data['cache_name'] = list(cache_name.keys())
                extra_data['cache_key'] = [n.split(':')[-1] for n in cache_name.keys()]
            else:
                extra_data['cache_key'] = cache_name

        with CaptureSpan(name, "cache.memcached", extra_data):
            return wrapped(*args, **kwargs)
