"""Feature Engineer module for enriching transaction data.

NOTE: This is legacy code preserved for future integration into statement_processor.
It provides stubs for description parsing that could be implemented as a preprocessing step.
"""

import logging
from typing import Optional

import pandas as pd
from pydantic import BaseModel, Field

logger = logging.getLogger(__name__)


class ParsedDescription(BaseModel):
    """Structured components parsed from a transaction description.
    
    Credit card descriptions typically follow the format:
    VENDOR_NAME [PHONE_NUMBER] [CITY] STATE
    
    Examples:
        - "WALMART.COM 8009256278 800-966-6546 AR"
        - "VCO*St. Brendan Cathol 770-205-7969 GA"
        - "THE HOME DEPOT #6978 SUWANEE GA"
    """
    
    vendor_name: str = Field(description="Extracted vendor/merchant name")
    phone_number: Optional[str] = Field(default=None, description="Phone number if present")
    city: Optional[str] = Field(default=None, description="City name if present")
    state: Optional[str] = Field(default=None, description="2-letter state code")
    raw_description: str = Field(description="Original description string")


class FeatureEngineer:
    """Enriches transaction DataFrames with engineered features.
    
    This class provides methods to parse transaction descriptions and add
    structured columns for vendor name, location, and other derived features.
    
    TODO: Implement description parsing to extract:
        - vendor_name: Clean vendor name without location/phone
        - phone_number: Extracted phone number (various formats)
        - city: City name from description
        - state: 2-letter state code
        - is_online: Whether transaction appears to be online (has .COM, www, etc.)
        - category: Inferred category (retail, food, utilities, etc.)
    """

    def __init__(self) -> None:
        """Initialize the feature engineer."""
        pass

    def enrich(self, df: pd.DataFrame) -> pd.DataFrame:
        """Add engineered features to transaction DataFrame.
        
        Args:
            df: DataFrame with at least 'description' column.
            
        Returns:
            DataFrame with additional feature columns.
            
        TODO: Implement feature extraction:
            - Parse description into vendor_name, phone, city, state
            - Add is_online flag
            - Add day_of_week, month from date
            - Add amount_category (small/medium/large)
        """
        if df.empty:
            return df
        
        logger.debug(f"Feature engineering stub called with {len(df)} rows")
        return df

    def parse_description(self, description: str) -> ParsedDescription:
        """Parse a transaction description into structured components.
        
        Args:
            description: Raw transaction description string.
            
        Returns:
            ParsedDescription with extracted components.
            
        TODO: Implement parsing logic:
            - Extract 2-letter state code from end
            - Extract phone numbers (XXX-XXX-XXXX, XXXXXXXXXX formats)
            - Extract city name (typically before state)
            - Remaining text is vendor name
        """
        return ParsedDescription(
            vendor_name=description,
            raw_description=description,
        )
