"""Statement Processor - Extensible PDF financial statement processing framework.

This package provides a plugin-based architecture for extracting transaction data
from PDF financial statements, parsing them into structured data, and clustering
transactions by vendor.

Core components:
- BaseStatementParser, BaseTransaction: Interfaces for parser plugins
- ClusteringStrategy, ClusterRunner: Analytics framework
- Plugin registry: Auto-discovers installed add-ons

Example usage:
    from statement_processor import StatementProcessor

    processor = StatementProcessor(
        input_dirs=["./statements"],
        output_dir="./results",
    )
    result = processor.process()
"""

__version__ = "0.1.0"


def __getattr__(name: str):
    """Lazy imports to avoid circular dependencies."""
    if name == "BaseTransaction":
        from statement_processor.core.base_transaction import BaseTransaction
        return BaseTransaction
    if name == "BaseStatementParser":
        from statement_processor.core.base_parser import BaseStatementParser
        return BaseStatementParser
    if name in ("discover_parsers", "discover_strategies", "auto_detect_parser"):
        from statement_processor.core import registry
        return getattr(registry, name)
    if name == "Transaction":
        from statement_processor.models import Transaction
        return Transaction
    if name == "Statement":
        from statement_processor.models import Statement
        return Statement
    if name == "ProcessingResult":
        from statement_processor.models import ProcessingResult
        return ProcessingResult
    if name == "StatementProcessor":
        from statement_processor.processor import StatementProcessor
        return StatementProcessor
    raise AttributeError(f"module {__name__!r} has no attribute {name!r}")


__all__ = [
    # Core interfaces
    "BaseTransaction",
    "BaseStatementParser",
    # Discovery
    "discover_parsers",
    "discover_strategies",
    "auto_detect_parser",
    # Models
    "Transaction",
    "Statement",
    "ProcessingResult",
    # Main processor
    "StatementProcessor",
]
