"""Core data models and interfaces for the clustering framework.

This module defines the base classes for transaction clustering:
- TransactionCluster: A group of transactions with membership scores
- ClusteringStrategy: Abstract base class for clustering algorithms
"""

from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from typing import Any, Dict, List

import pandas as pd


@dataclass
class TransactionCluster:
    """A group of transactions identified as belonging together.

    Each transaction has a membership score indicating confidence it belongs
    to this cluster. This allows fuzzy membership where some transactions
    are strong matches (1.0) and others are weaker matches (0.7).

    Attributes:
        memberships: Map of transaction index → confidence (0.0 to 1.0).
        label: Type of cluster (e.g., "vendor", "recurring_monthly").
        metadata: Strategy-specific details about the cluster.
    """

    memberships: Dict[int, float]
    label: str
    metadata: Dict[str, Any] = field(default_factory=dict)

    def __post_init__(self) -> None:
        """Validate all membership scores are within bounds."""
        for idx, conf in self.memberships.items():
            if not 0.0 <= conf <= 1.0:
                raise ValueError(
                    f"membership score must be between 0.0 and 1.0, got {conf} for index {idx}"
                )

    @property
    def indices(self) -> List[int]:
        """Return all transaction indices in this cluster.

        Returns:
            List of transaction indices that belong to this cluster.
        """
        return list(self.memberships.keys())

    def indices_above_threshold(self, threshold: float) -> List[int]:
        """Return indices with membership >= threshold.

        Args:
            threshold: Minimum membership score to include.

        Returns:
            List of transaction indices with membership at or above threshold.
        """
        return [idx for idx, conf in self.memberships.items() if conf >= threshold]


class ClusteringStrategy(ABC):
    """Abstract base class for transaction clustering strategies.

    Implementations analyze a DataFrame of transactions and return
    clusters grouping related transactions together.

    Third-party plugins implement this interface and register via entry points:

        [project.entry-points."statement_processor.strategies"]
        my_strategy = "my_package:MyStrategy"
    """

    @property
    @abstractmethod
    def name(self) -> str:
        """Return the strategy name.

        Returns:
            A unique identifier for this strategy.
        """

    @abstractmethod
    def cluster(self, transactions: pd.DataFrame) -> List[TransactionCluster]:
        """Group transactions into clusters.

        Args:
            transactions: DataFrame with columns: date, description, amount.

        Returns:
            List of TransactionCluster objects. Empty list if no clusters found.
        """
