"""ExactMatchStrategy for grouping transactions by identical description.

This strategy creates clusters where all transactions with the same
description are grouped together with 1.0 membership confidence.
"""

import logging
from typing import List

import pandas as pd

from statement_processor.analytics.clustering import ClusteringStrategy, TransactionCluster

logger = logging.getLogger(__name__)


class ExactMatchStrategy(ClusteringStrategy):
    """Groups transactions with identical descriptions.

    This is a simple reference implementation that creates vendor clusters
    by grouping transactions that have exactly the same description string.
    All matches receive a membership score of 1.0.
    """

    @property
    def name(self) -> str:
        """Return the strategy name.

        Returns:
            The string "exact_match".
        """
        return "exact_match"

    def cluster(self, transactions: pd.DataFrame) -> List[TransactionCluster]:
        """Group transactions by identical description.

        Args:
            transactions: DataFrame with columns: date, description, amount.

        Returns:
            List of TransactionCluster objects, one per unique description.
            Empty list if no transactions or no valid clusters found.
        """
        if transactions.empty:
            return []

        clusters: List[TransactionCluster] = []

        try:
            for description, group in transactions.groupby("description"):
                memberships = {idx: 1.0 for idx in group.index}
                clusters.append(
                    TransactionCluster(
                        memberships=memberships,
                        label=f"vendor:{description}",
                        metadata={"vendor_name": description, "match_strategy": "exact"},
                    )
                )
        except Exception as e:
            logger.error(f"ExactMatchStrategy error: {e}")
            return []

        return clusters
