"""Data model for vendor clustering results.

This module defines the VendorCluster dataclass that represents a group
of transactions identified as belonging to the same vendor/merchant.
"""

from dataclasses import dataclass
from typing import List


@dataclass
class VendorCluster:
    """A group of transactions identified as belonging to the same vendor.

    Attributes:
        vendor_name: Representative name for the cluster (e.g., "AMAZON.COM").
        transaction_indices: DataFrame indices of transactions in this cluster.
        transaction_count: Number of transactions in the cluster.
        total_amount: Sum of all transaction amounts in the cluster.
        earliest_date: First transaction date (YYYY-MM-DD format).
        latest_date: Most recent transaction date (YYYY-MM-DD format).
    """

    vendor_name: str
    transaction_indices: List[int]
    transaction_count: int
    total_amount: float
    earliest_date: str
    latest_date: str

    def __post_init__(self) -> None:
        """Validate cluster data consistency."""
        if self.transaction_count != len(self.transaction_indices):
            raise ValueError(
                f"transaction_count ({self.transaction_count}) must equal "
                f"length of transaction_indices ({len(self.transaction_indices)})"
            )
        if self.transaction_count < 1:
            raise ValueError("transaction_count must be at least 1")
