"""Vendor regex patterns for canonical name mapping.

This package auto-loads vendor patterns from YAML files in this directory.
Patterns are matched against transaction descriptions (case-insensitive).

Third-party plugins can provide additional patterns via entry points or
by passing extra directories to load_patterns().

YAML file format::

    patterns:
      - pattern: 'VENDOR_NAME\\*.*'
        vendor: Canonical Vendor Name
"""

import logging
import re
from pathlib import Path
from typing import Any, Dict, List, Optional

import yaml

logger = logging.getLogger(__name__)


def load_patterns(extra_dirs: Optional[List[Path]] = None) -> Dict[str, str]:
    """Load vendor patterns from YAML files.

    Loads patterns from the built-in directory and any additional
    directories provided. This allows third-party plugins and users
    to add custom patterns.

    Args:
        extra_dirs: Optional list of additional directories to load from.

    Returns:
        Dict mapping regex patterns to canonical vendor names.
    """
    patterns: Dict[str, str] = {}
    
    # Start with built-in patterns directory
    dirs_to_scan = [Path(__file__).parent]
    
    # Add any extra directories
    if extra_dirs:
        dirs_to_scan.extend(extra_dirs)

    for patterns_dir in dirs_to_scan:
        if not patterns_dir.exists():
            logger.warning(f"Patterns directory not found: {patterns_dir}")
            continue
            
        for yaml_file in sorted(patterns_dir.glob("*.yaml")):
            try:
                with open(yaml_file, encoding="utf-8") as f:
                    data: Dict[str, Any] = yaml.safe_load(f) or {}

                file_patterns = data.get("patterns", [])
                for entry in file_patterns:
                    pattern = entry.get("pattern")
                    vendor = entry.get("vendor")
                    if pattern and vendor:
                        # Validate regex
                        try:
                            re.compile(pattern)
                            patterns[pattern] = vendor
                        except re.error as e:
                            logger.warning(
                                f"Invalid regex in {yaml_file.name}: '{pattern}' - {e}"
                            )

                logger.debug(f"Loaded {len(file_patterns)} patterns from {yaml_file.name}")

            except Exception as e:
                logger.error(f"Failed to load {yaml_file.name}: {e}")

    return patterns


def _load_yaml_patterns() -> Dict[str, str]:
    """Auto-discover and load all YAML pattern files from built-in directory.

    Returns:
        Dict mapping regex patterns to canonical vendor names.
    """
    return load_patterns()


# Pre-load built-in patterns
VENDOR_PATTERNS: Dict[str, str] = _load_yaml_patterns()

__all__ = ["VENDOR_PATTERNS", "load_patterns"]
