#!/usr/bin/env python
"""Statement Processor CLI - Command-line interface.

Usage:
    statement-processor <input_dir> [<input_dir2> ...] [--output <dir>] [--debug <dir>]

Examples:
    statement-processor ./statements
    statement-processor ./2024_statements ./2025_statements --output ./results
    statement-processor ./statements --debug ./debug_output
"""

import argparse
import logging
import sys
from pathlib import Path

from statement_processor.processor import StatementProcessor


def setup_logging(debug_dir: Path | None = None) -> None:
    """Configure logging with optional file output."""
    import os
    import warnings

    os.environ["TRANSFORMERS_VERBOSITY"] = "info"
    warnings.filterwarnings(
        "ignore", message="The extract API is currently experimental"
    )

    root_logger = logging.getLogger()
    root_logger.handlers.clear()
    root_logger.setLevel(logging.INFO)

    formatter = logging.Formatter("%(asctime)s - %(levelname)s - %(message)s")

    stdout_handler = logging.StreamHandler(sys.stdout)
    stdout_handler.setFormatter(formatter)
    root_logger.addHandler(stdout_handler)

    if debug_dir:
        debug_dir.mkdir(parents=True, exist_ok=True)
        file_handler = logging.FileHandler(debug_dir / "processing.log")
        file_handler.setFormatter(formatter)
        root_logger.addHandler(file_handler)

    logging.getLogger("docling").setLevel(logging.WARNING)
    logging.getLogger("rapidocr").setLevel(logging.ERROR)
    logging.getLogger("transformers").setLevel(logging.ERROR)


def main() -> int:
    """Main entry point."""
    parser = argparse.ArgumentParser(
        description="Process financial statement PDFs and generate vendor summary.",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog=__doc__,
    )
    parser.add_argument(
        "input_dirs",
        nargs="+",
        type=Path,
        help="Input directories containing PDF statements",
    )
    parser.add_argument(
        "--output",
        "-o",
        type=Path,
        default=Path("."),
        help="Output directory for CSV files (default: current directory)",
    )
    parser.add_argument(
        "--debug",
        "-d",
        type=Path,
        default=None,
        help="Debug directory for extracted text and logs",
    )
    parser.add_argument(
        "--parser",
        "-p",
        type=str,
        default=None,
        help="Parser name to use (auto-detect if not specified)",
    )
    parser.add_argument(
        "--no-strict",
        action="store_true",
        default=False,
        help="Disable strict validation (continue on transaction sum mismatch)",
    )
    parser.add_argument(
        "--tolerance",
        "-t",
        type=float,
        default=0.0,
        help="Tolerance for validation comparisons in dollars (default: 0.0)",
    )

    args = parser.parse_args()

    setup_logging(args.debug)
    logger = logging.getLogger(__name__)

    logger.info("Starting Statement Processor")
    logger.info(f"Input directories: {args.input_dirs}")
    logger.info(f"Output directory: {args.output}")

    processor = StatementProcessor(
        input_dirs=[str(d) for d in args.input_dirs],
        output_dir=str(args.output),
        debug_dir=str(args.debug) if args.debug else None,
        parser_name=args.parser,
        strict=not args.no_strict,
        tolerance=args.tolerance,
    )

    result = processor.process()

    if not result.total_transactions:
        logger.warning("No transactions found!")
        return 1

    # Print summary
    print("\n" + "=" * 60)
    print("Processing Summary")
    print("=" * 60)
    print(f"Total files: {result.total_files}")
    print(f"Successful: {result.successful_files}")
    print(f"Failed: {result.failed_files}")
    print(f"Total transactions: {result.total_transactions}")

    if result.errors:
        print(f"\nErrors ({len(result.errors)}):")
        for error in result.errors[:5]:
            print(f"  - {error}")
        if len(result.errors) > 5:
            print(f"  ... and {len(result.errors) - 5} more")

    print(f"\nOutput files:")
    print(f"  {args.output / 'vendor_amounts.csv'}")
    print("=" * 60)

    return 0


if __name__ == "__main__":
    sys.exit(main())
