"""Base parser interface for Statement Processor.

Parser plugins implement this interface to add support for
specific statement formats (banks, credit cards, etc.).
"""

from abc import ABC, abstractmethod
from typing import List

from statement_processor.core.base_transaction import BaseTransaction


class BaseStatementParser(ABC):
    """Abstract base class for statement parsers.

    Implementations handle specific statement formats by:
    1. Detecting if they can parse a given document (can_parse)
    2. Extracting transactions from the document (parse)

    Parsers are discovered via entry points and auto-selected
    based on document content.

    Example implementation:
        class ChaseParser(BaseStatementParser):
            @property
            def name(self) -> str:
                return "chase_sapphire"

            def can_parse(self, raw_text: str) -> bool:
                return "Chase" in raw_text and "Sapphire" in raw_text

            def parse(self, raw_text: str, source_file: str) -> list[BaseTransaction]:
                # Chase-specific parsing logic
                ...
    """

    @property
    @abstractmethod
    def name(self) -> str:
        """Unique identifier for this parser.

        Used for logging, selection, and entry point registration.

        Returns:
            Parser name string (e.g., 'chase_sapphire', 'amex_gold').
        """

    @abstractmethod
    def can_parse(self, raw_text: str) -> bool:
        """Check if this parser can handle the given document.

        Called during auto-detection to find the appropriate parser.
        Should check for format-specific markers in the text.

        Args:
            raw_text: Extracted text content from the PDF.

        Returns:
            True if this parser can handle the document.
        """

    @abstractmethod
    def parse(self, raw_text: str, source_file: str) -> List[BaseTransaction]:
        """Parse document text into transactions.

        Args:
            raw_text: Extracted text content from the PDF.
            source_file: Original filename for tracking/logging.

        Returns:
            List of transactions extracted from the document.
        """

    def parse_with_metadata(
        self, raw_text: str, source_file: str
    ) -> tuple[List[BaseTransaction], dict]:
        """Parse document and return transactions with metadata.

        Optional method for parsers that extract additional metadata
        (account info, statement period, etc.).

        Default implementation returns empty metadata.

        Args:
            raw_text: Extracted text content from the PDF.
            source_file: Original filename for tracking/logging.

        Returns:
            Tuple of (transactions, metadata_dict).
        """
        return self.parse(raw_text, source_file), {}
