"""Base transaction interface for Statement Processor.

All transaction types must implement this minimal interface to be
compatible with the analytics framework.
"""

from abc import ABC, abstractmethod
from typing import Any, Dict


class BaseTransaction(ABC):
    """Minimal transaction interface.

    All transaction implementations must provide date, description, and amount.
    Additional fields can be added by specific implementations.

    The analytics framework only requires these three fields, so any
    statement format can be supported as long as it can extract these.
    """

    @property
    @abstractmethod
    def date(self) -> str:
        """Transaction date in ISO format (YYYY-MM-DD).

        Returns:
            Date string in YYYY-MM-DD format.
        """

    @property
    @abstractmethod
    def description(self) -> str:
        """Transaction description or merchant name.

        Returns:
            Description string.
        """

    @property
    @abstractmethod
    def amount(self) -> float:
        """Transaction amount.

        Positive values typically represent charges/debits.
        Negative values typically represent credits/refunds.

        Returns:
            Amount as float.
        """

    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for DataFrame creation.

        Default implementation returns the three core fields.
        Subclasses can override to include additional fields.

        Returns:
            Dictionary with at least date, description, amount keys.
        """
        return {
            "date": self.date,
            "description": self.description,
            "amount": self.amount,
        }
