"""Plugin registry for discovering parsers and strategies.

Uses Python entry points for automatic discovery of installed plugins.
Third-party packages register their plugins in pyproject.toml:

    [project.entry-points."statement_processor.parsers"]
    my_parser = "my_package:MyParser"

    [project.entry-points."statement_processor.strategies"]
    my_strategy = "my_package:MyStrategy"
"""

import logging
from importlib.metadata import entry_points
from typing import List, Optional, Type

from statement_processor.core.base_parser import BaseStatementParser

logger = logging.getLogger(__name__)

# Entry point group names
PARSERS_GROUP = "statement_processor.parsers"
STRATEGIES_GROUP = "statement_processor.strategies"


def discover_parsers() -> List[Type[BaseStatementParser]]:
    """Discover all installed parser plugins.

    Loads parser classes from entry points registered under
    the 'statement_processor.parsers' group.

    Returns:
        List of parser classes (not instances).
    """
    parser_classes: List[Type[BaseStatementParser]] = []

    try:
        eps = entry_points(group=PARSERS_GROUP)
        for ep in eps:
            try:
                parser_cls = ep.load()
                if isinstance(parser_cls, type) and issubclass(
                    parser_cls, BaseStatementParser
                ):
                    parser_classes.append(parser_cls)
                    logger.debug(f"Discovered parser: {ep.name} -> {parser_cls}")
                else:
                    logger.warning(
                        f"Entry point {ep.name} is not a BaseStatementParser subclass"
                    )
            except Exception as e:
                logger.error(f"Failed to load parser entry point {ep.name}: {e}")
    except Exception as e:
        logger.error(f"Failed to discover parsers: {e}")

    # Always include built-in parsers
    from statement_processor.parsers.markdown_table import MarkdownTableParser

    if MarkdownTableParser not in parser_classes:
        parser_classes.append(MarkdownTableParser)

    return parser_classes


def discover_strategies() -> List[Type]:
    """Discover all installed strategy plugins.

    Loads strategy classes from entry points registered under
    the 'statement_processor.strategies' group.

    Returns:
        List of strategy classes (not instances).
    """
    from statement_processor.analytics.clustering import ClusteringStrategy

    strategy_classes: List[Type[ClusteringStrategy]] = []

    try:
        eps = entry_points(group=STRATEGIES_GROUP)
        for ep in eps:
            try:
                strategy_cls = ep.load()
                if isinstance(strategy_cls, type) and issubclass(
                    strategy_cls, ClusteringStrategy
                ):
                    strategy_classes.append(strategy_cls)
                    logger.debug(f"Discovered strategy: {ep.name} -> {strategy_cls}")
                else:
                    logger.warning(
                        f"Entry point {ep.name} is not a ClusteringStrategy subclass"
                    )
            except Exception as e:
                logger.error(f"Failed to load strategy entry point {ep.name}: {e}")
    except Exception as e:
        logger.error(f"Failed to discover strategies: {e}")

    return strategy_classes


def auto_detect_parser(raw_text: str) -> Optional[BaseStatementParser]:
    """Find a parser that can handle the given document.

    Iterates through all discovered parsers and returns the first
    one that reports it can parse the document.

    Args:
        raw_text: Extracted text content from the PDF.

    Returns:
        Parser instance if found, None otherwise.
    """
    for parser_cls in discover_parsers():
        try:
            parser = parser_cls()
            if parser.can_parse(raw_text):
                logger.info(f"Auto-detected parser: {parser.name}")
                return parser
        except Exception as e:
            logger.warning(f"Error checking parser {parser_cls}: {e}")

    logger.warning("No parser found for document")
    return None


def get_parser_by_name(name: str) -> Optional[BaseStatementParser]:
    """Get a specific parser by name.

    Args:
        name: Parser name to find.

    Returns:
        Parser instance if found, None otherwise.
    """
    for parser_cls in discover_parsers():
        try:
            parser = parser_cls()
            if parser.name == name:
                return parser
        except Exception as e:
            logger.warning(f"Error instantiating parser {parser_cls}: {e}")

    return None
