"""PDF Scanner module for discovering PDF files in directories."""

import logging
from pathlib import Path
from typing import List

logger = logging.getLogger(__name__)


class PDFScanner:
    """Scans directories to discover PDF files.

    Attributes:
        input_directory: Path to the directory to scan for PDF files.
    """

    def __init__(self, input_directory: str):
        """Initialize scanner with input directory path.

        Args:
            input_directory: Path to the directory containing PDF files.
        """
        self.input_directory = Path(input_directory)

    def validate_directory(self) -> bool:
        """Check if directory exists and is readable.

        Returns:
            True if directory exists and is accessible, False otherwise.
        """
        try:
            return self.input_directory.exists() and self.input_directory.is_dir()
        except (OSError, PermissionError) as e:
            logger.error(f"Cannot access directory {self.input_directory}: {e}")
            return False

    def scan(self) -> List[Path]:
        """Return list of PDF file paths found in directory.

        Scans the input directory (non-recursively) for files with .pdf extension.

        Returns:
            List of Path objects for all PDF files found.
            Returns empty list if directory is invalid or contains no PDFs.
        """
        if not self.validate_directory():
            logger.warning(f"Directory {self.input_directory} is not valid")
            return []

        pdf_files: List[Path] = []
        try:
            for item in self.input_directory.iterdir():
                if item.is_file() and item.suffix.lower() == ".pdf":
                    pdf_files.append(item)
        except (OSError, PermissionError) as e:
            logger.error(f"Error scanning directory {self.input_directory}: {e}")
            return []

        logger.info(f"Found {len(pdf_files)} PDF files in {self.input_directory}")
        return pdf_files
