"""
StrateQueue REST Daemon

Minimal FastAPI server that provides REST endpoints for the Web UI.
Handles configuration management (reading/writing credentials.env).
"""
import sys, pathlib
# Add project root's 'src' directory to path to allow absolute imports
SRC_ROOT = pathlib.Path(__file__).resolve().parents[2]
if str(SRC_ROOT) not in sys.path:
    sys.path.insert(0, str(SRC_ROOT))

from pathlib import Path
from typing import Dict, Any

from fastapi import FastAPI, HTTPException, UploadFile, File
from pydantic import BaseModel
from fastapi.middleware.cors import CORSMiddleware
import subprocess
import asyncio
from asyncio.subprocess import PIPE, STDOUT
import uuid
import re
import json
import logging
import signal
from datetime import datetime, timezone
from fastapi import Body
import socket, contextlib, httpx  # new imports for stats

CRED_FILE = Path.home() / ".stratequeue" / "credentials.env"

app = FastAPI(title="StrateQueue Daemon", version="1.0.0")

# Enable CORS so the Vite dev server (typically http://localhost:5173) can access the API
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],  # in dev we allow all; tighten in prod if needed
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)


# ───────────────────────── helpers ─────────────────────────
def _read_env() -> dict[str, str]:
    """Read environment variables from credentials file."""
    if not CRED_FILE.exists():
        return {}
    
    kv = {}
    for line in CRED_FILE.read_text().splitlines():
        line = line.strip()
        if line and not line.startswith("#") and "=" in line:
            k, v = line.split("=", 1)
            kv[k.strip()] = v.strip()
    return kv


def _write_env(new: dict[str, str]) -> None:
    """Write environment variables to credentials file, preserving existing ones."""
    env = _read_env()
    env.update(new)

    CRED_FILE.parent.mkdir(exist_ok=True)
    body = ["# StrateQueue Credentials", "# Generated by daemon", ""]
    body += [f"{k}={v}" for k, v in env.items()]
    CRED_FILE.write_text("\n".join(body) + "\n")


# ──────────────────────── models ───────────────────────────
class ConfigPayload(BaseModel):
    """Payload for configuration updates - arbitrary key/value pairs."""
    class Config:
        extra = "allow"  # Allow arbitrary fields
    
    def dict(self, **kwargs) -> Dict[str, Any]:
        """Override to return all fields including extra ones."""
        return super().dict(**kwargs)


# ──────────────────────── routes ───────────────────────────
@app.get("/health")
def health():
    """Health check endpoint."""
    return {"status": "ok"}


@app.get("/engines")
def get_engines():
    """Get information about available and unavailable engines."""
    try:
        from StrateQueue.engines.engine_factory import (
            get_supported_engines, 
            get_unavailable_engines,
            get_all_known_engines
        )
        
        supported = get_supported_engines()
        unavailable = get_unavailable_engines()
        all_known = get_all_known_engines()
        
        # Build response with engine availability information
        engines = []
        for engine_name in all_known:
            if engine_name in supported:
                engines.append({
                    "name": engine_name,
                    "available": True,
                    "reason": None
                })
            else:
                engines.append({
                    "name": engine_name,
                    "available": False,
                    "reason": unavailable.get(engine_name, "Unknown reason")
                })
        
        return {
            "engines": engines,
            "supported_count": len(supported),
            "total_count": len(all_known)
        }
    except Exception as e:
        return {
            "engines": [],
            "supported_count": 0,
            "total_count": 0,
            "error": str(e)
        }


@app.get("/config")
def get_config():
    """Get current configuration from credentials file."""
    return _read_env()


@app.post("/config")
def save_config(payload: Dict[str, str]):
    """Save configuration to credentials file."""
    if not payload:
        raise HTTPException(400, "Empty payload")
    
    # Filter out empty values
    filtered_payload = {k: v for k, v in payload.items() if v and v.strip()}
    
    if not filtered_payload:
        raise HTTPException(400, "No valid configuration provided")
    
    _write_env(filtered_payload)
    return {
        "message": "Configuration saved successfully", 
        "saved_keys": list(filtered_payload.keys())
    }


# ──────────────────────── file upload ───────────────────────────

@app.post("/upload_strategy")
async def upload_strategy(file: UploadFile = File(...)):
    """Save uploaded strategy file to a temp directory and return its path."""
    try:
        import tempfile, shutil
        dest_dir = Path.home() / ".stratequeue" / "uploaded_strategies"
        dest_dir.mkdir(parents=True, exist_ok=True)
        
        # Use original filename, but handle conflicts by adding a number
        original_name = Path(file.filename).name if file.filename else "strategy.py"
        dest_path = dest_dir / original_name
        
        # If file exists, add a number suffix
        counter = 1
        while dest_path.exists():
            stem = Path(original_name).stem
            suffix = Path(original_name).suffix or ".py"
            dest_path = dest_dir / f"{stem}_{counter}{suffix}"
            counter += 1
        
        # Save the file
        with dest_path.open("wb") as fout:
            shutil.copyfileobj(file.file, fout)
        
        saved_path = dest_path.resolve()

        return {"path": str(saved_path)}
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))


# ──────────────────────── deploy endpoints ───────────────────────────
running_systems: dict[str, dict] = {}

# Simple regex patterns to parse CLI output
SIG_RE = re.compile(r"Extracted signal:\s+(\w+).*")
TRADE_RE = re.compile(r"(BUY|SELL).*@\s*([-+]?\d+\.\d+)")

log = logging.getLogger(__name__)

async def monitor_strategy_output(job_id: str, proc: subprocess.Popen):
    """Background task to read CLI output and update strategy metadata."""
    try:
        while proc.returncode is None:
            line = await proc.stdout.readline()
            if not line:  # EOF
                await asyncio.sleep(0.1)
                continue

            line = line.decode(errors='ignore').strip()
            if not line:
                continue
                
            # Update metadata based on parsed output
            if job_id in running_systems:
                meta = running_systems[job_id]["meta"]
                
                # Parse signal extractions
                if sig_match := SIG_RE.search(line):
                    signal_type = sig_match.group(1)
                    meta["last_signal"] = datetime.now(timezone.utc).isoformat()
                    meta["last_signal_type"] = signal_type
                    log.info(f"Strategy {job_id}: Signal {signal_type}")
                
                # Parse trade executions
                elif trade_match := TRADE_RE.search(line):
                    trade_type = trade_match.group(1)
                    price = float(trade_match.group(2))
                    meta["last_signal"] = datetime.now(timezone.utc).isoformat()
                    meta["last_signal_type"] = trade_type
                    log.info(f"Strategy {job_id}: {trade_type} @ ${price:.2f}")
                    
    except Exception as e:
        log.error(f"Error monitoring strategy {job_id}: {e}")
    finally:
        # Mark as finished when process exits
        if job_id in running_systems:
            running_systems[job_id]["meta"]["status"] = "finished"

@app.post("/deploy/validate")
async def validate_deploy(_: Dict[str, Any]):
    return {"valid": True, "errors": [], "warnings": []}


@app.post("/deploy/start")
async def start_deploy(payload: Dict[str, Any]):
    """
    Launch `stratequeue deploy …` in a subprocess and monitor its output for live stats.
    """
    # Choose a free port for the worker's stats server
    stats_port = _get_free_port()

    # Build stratequeue CLI command ---------------------------------
    cmd: list[str] = [
        "stratequeue", "deploy",
        "--strategy", payload["strategy"],
        "--symbol", payload.get("symbol", "AAPL"),
        "--data-source", payload.get("data_source", "demo"),
        "--timeframe", payload.get("granularity", "1m"),
        "--lookback", str(payload.get("lookback", 1000)),
        "--duration", str(payload.get("duration", 60)),
        "--stats-port", str(stats_port),
    ]

    if payload.get("strategy_id"):
        cmd += ["--strategy-id", str(payload["strategy_id"])]
    if payload.get("allocation"):
        cmd += ["--allocation", str(payload["allocation"])]
    if payload.get("broker"):
        cmd += ["--broker", payload["broker"]]
    if payload.get("engine"):
        cmd += ["--engine", payload["engine"]]

    mode = payload.get("mode", "signals")
    if mode == "paper":
        cmd.append("--paper")
    elif mode == "live":
        cmd.append("--live")
    else:
        cmd.append("--no-trading")

    # ---------------------------------------------------------------
    job_id = str(uuid.uuid4())

    # Spawn subprocess asynchronously so event-loop is not blocked
    proc = await asyncio.create_subprocess_exec(
        *cmd,
        stdout=PIPE,
        stderr=STDOUT,
    )

    # Store metadata and process info
    strategy_name = Path(payload["strategy"]).stem
    running_systems[job_id] = {
        "proc": proc,
        "meta": {
            "id": job_id,
            "name": strategy_name,
            "symbol": payload.get("symbol", "AAPL"),
            "status": "running",
            "mode": mode,
            "data_source": payload.get("data_source", "demo"),
            "granularity": payload.get("granularity", "1m"),
            "allocation": float(payload.get("allocation", 0) or 0),
            "pnl": 0,
            "pnl_percent": 0,
            "started_at": datetime.now(timezone.utc).isoformat(),
            "last_signal": None,
            "last_signal_type": None,
            "stats_url": f"http://127.0.0.1:{stats_port}/stats",
        },
    }

    # Start background task to monitor output
    asyncio.create_task(monitor_strategy_output(job_id, proc))
    
    log.info(f"Spawned strategy {job_id} - PID {proc.pid}")
    return {"status": "started", "job_id": job_id, "cmd": cmd}


# ──────────────────────── strategy status endpoints ───────────────────────────

@app.get("/strategies")
async def get_strategies():
    """
    List running strategies with live stats parsed from CLI output.
    """
    out: list[dict] = []
    for jid, info in list(running_systems.items()):
        proc = info["proc"]
        meta = info["meta"]

        # Update status from subprocess
        if proc.returncode is None:
            meta["status"] = "running"
        else:
            meta["status"] = "finished"
            
        out.append(meta)

        # Clean up finished jobs to keep dict small
        if proc.returncode is not None:
            running_systems.pop(jid, None)

    return {"strategies": out}


@app.get("/strategies/{strategy_id}")
async def get_strategy_details(strategy_id: str):
    """Get detailed information about a specific strategy."""
    try:
        # Check if strategy exists in running systems
        if strategy_id in running_systems:
            strategy_info = running_systems[strategy_id]
            proc = strategy_info["proc"]
            meta = strategy_info["meta"]
            
            # Update process status
            if proc.returncode is None:
                meta["status"] = "running"
            else:
                meta["status"] = "finished"
            
            return meta
        else:
            raise HTTPException(status_code=404, detail="Strategy not found")
    except Exception as e:
        raise HTTPException(status_code=404, detail=f"Strategy not found: {e}")


@app.get("/strategies/{strategy_id}/statistics")
async def get_strategy_statistics(strategy_id: str):
    """Get basic statistics for a specific strategy."""
    try:
        if strategy_id not in running_systems:
            raise HTTPException(status_code=404, detail="Strategy not found")
        
        strategy_info = running_systems[strategy_id]
        meta = strategy_info["meta"]
        
        stats_url = meta.get("stats_url")
        stats_data = None
        if stats_url:
            try:
                resp = httpx.get(stats_url, timeout=2.0)
                if resp.status_code == 200:
                    stats_data = resp.json()
            except Exception as e:
                log.warning(f"Failed to fetch stats from {stats_url}: {e}")

        return {
            "strategy_id": strategy_id,
            "stats_url": stats_url,
            "metrics": stats_data,
            "captured_at": datetime.now(timezone.utc).isoformat(),
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error retrieving statistics: {e}")


@app.post("/strategies/{strategy_id}/pause")
async def pause_strategy(strategy_id: str):
    """Pause a running strategy."""
    try:
        # In a real implementation, this would pause the actual strategy
        return {"message": f"Strategy {strategy_id} paused successfully"}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error pausing strategy: {str(e)}")


@app.post("/strategies/{strategy_id}/resume")
async def resume_strategy(strategy_id: str):
    """Resume a paused strategy."""
    try:
        # In a real implementation, this would resume the actual strategy
        return {"message": f"Strategy {strategy_id} resumed successfully"}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error resuming strategy: {str(e)}")


@app.post("/strategies/{strategy_id}/stop")
async def stop_strategy(strategy_id: str, opts: dict = Body(default={})):
    """Stop a running strategy."""
    try:
        if strategy_id not in running_systems:
            raise HTTPException(status_code=404, detail="Strategy not found")

        liquidate = bool(opts.get("liquidate", False))
        force = bool(opts.get("force", False))
        
        GRACE_SECONDS = 5

        entry = running_systems[strategy_id]
        proc = entry["proc"]
        
        # Mark as stopping
        entry["meta"]["status"] = "stopping"

        # 1) Optional liquidation (best-effort, don't block forever)
        if liquidate:
            try:
                # For now, we'll log the liquidation request
                # In a full implementation, this would integrate with the broker
                # to close all positions for this strategy
                log.info(f"Liquidation requested for strategy {strategy_id}")
                # TODO: Implement actual position liquidation via broker integration
            except Exception as e:
                log.warning(f"Liquidation failed for {strategy_id}: {e}")

        # 2) Graceful shutdown
        if proc.returncode is None:
            proc.send_signal(signal.SIGINT)  # Graceful shutdown
            try:
                await asyncio.wait_for(proc.wait(), timeout=GRACE_SECONDS)
                log.info(f"Strategy {strategy_id} stopped gracefully")
            except asyncio.TimeoutError:
                if force:
                    proc.kill()  # Force kill
                    await proc.wait()
                    log.info(f"Strategy {strategy_id} force killed")
                else:
                    log.warning(f"Strategy {strategy_id} did not stop gracefully within {GRACE_SECONDS}s")

        # 3) Final bookkeeping
        entry["meta"]["status"] = "stopped"
        
        return {
            "stopped": True,
            "liquidated": liquidate,
            "message": f"Strategy {strategy_id} stopped" + (" (forced)" if force else "")
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error stopping strategy: {str(e)}")

# Helper to find free port
def _get_free_port() -> int:
    with contextlib.closing(socket.socket(socket.AF_INET, socket.SOCK_STREAM)) as s:
        s.bind(('', 0))
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        return s.getsockname()[1] 