"""
Execution engine - builds iterator pipeline from logical plan.
"""

from .planner import LogicalPlan, JoinInfo
from .operators import (
    ScanIterator,
    FilterIterator,
    ProjectIterator,
    LookupJoinIterator,
    MergeJoinIterator
)


def execute_plan(
    plan,
    sources,
    source_metadata,
    debug=False
):
    """
    Execute a logical plan and return a generator of result rows.
    
    Args:
        plan: Logical execution plan
        sources: Dictionary mapping table names to source functions
        source_metadata: Dictionary with metadata about sources (e.g., ordered_by)
        
    Returns:
        Generator of result row dictionaries
    """
    # Start with scan of root table
    if debug:
        print(f"  [SCAN] Scanning table: {plan.root_table}")
    iterator = ScanIterator(
        sources[plan.root_table],
        plan.root_table,
        plan.root_alias,
        debug=debug
    )
    
    # Apply WHERE filter if present
    if plan.where_expr:
        if debug:
            print(f"  [FILTER] Applying WHERE clause")
        iterator = FilterIterator(iterator, plan.where_expr, debug=debug)
    
    # Apply joins in order
    for i, join_info in enumerate(plan.joins, 1):
        if debug:
            print(f"  [JOIN {i}/{len(plan.joins)}] {join_info.join_type} JOIN {join_info.table}")
        iterator = _build_join_iterator(
            iterator,
            join_info,
            sources,
            source_metadata,
            debug=debug
        )
    
    # Apply projection
    if debug:
        print(f"  [PROJECT] Applying SELECT projection")
        print(f"\nPipeline ready. Starting row processing...\n")
        print("-" * 60)
    iterator = ProjectIterator(iterator, plan.projections, debug=debug)
    
    return iterator
    
    
def _build_join_iterator(
    left_iterator,
    join_info,
    sources,
    source_metadata,
    debug=False
):
    """Build appropriate join iterator based on source capabilities."""
    right_source = sources[join_info.table]
    right_metadata = source_metadata.get(join_info.table, {})
    
    # Check if both sides are ordered by join keys
    left_ordered_by = _extract_table_from_key(join_info.left_key)
    right_ordered_by = right_metadata.get("ordered_by")
    
    # For merge join, we need both sides sorted on their respective join keys
    # This is a simplified check - in practice, we'd need to verify the actual
    # column names match
    use_merge_join = (
        right_ordered_by is not None and
        right_ordered_by == _extract_column_from_key(join_info.right_key)
    )
    
    if use_merge_join:
        if debug:
            print(f"      Using MERGE JOIN (sorted data)")
        return MergeJoinIterator(
            left_iterator,
            right_source,
            join_info.left_key,
            join_info.right_key,
            join_info.join_type,
            join_info.table,
            join_info.alias,
            debug=debug
        )
    else:
        if debug:
            print(f"      Using LOOKUP JOIN (building index...)")
        return LookupJoinIterator(
            left_iterator,
            right_source,
            join_info.left_key,
            join_info.right_key,
            join_info.join_type,
            join_info.table,
            join_info.alias,
            debug=debug
        )


def _extract_table_from_key(key):
    """Extract table alias from a key like 'alias.column'."""
    if "." in key:
        return key.split(".", 1)[0]
    return None


def _extract_column_from_key(key):
    """Extract column name from a key like 'alias.column'."""
    if "." in key:
        return key.split(".", 1)[1]
    return key

