import json
import logging
import time
import sys
from collections import defaultdict
from sqlglot import parse_one

# Optional: Memory monitoring
try:
    import psutil
    PSUTIL_AVAILABLE = True
except ImportError:
    PSUTIL_AVAILABLE = False

# Optional: Polars for batch writing
try:
    import polars as pl
    POLARS_AVAILABLE = True
except ImportError:
    POLARS_AVAILABLE = False

logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

try:
    from streaming_sql_engine import Engine
    from streaming_sql_engine import evaluator
    from sqlglot import expressions as exp
except ImportError:
    logger.error("streaming-sql-engine not installed. Run: pip install streaming-sql-engine --break-system-packages")
    exit(1)

_original_evaluate = evaluator.evaluate_expression

def patched_evaluate_expression(expr, row):
    expr_type = type(expr)
    
    if expr_type is exp.Literal:
        value = expr.this
        if isinstance(value, str):
            try:
                return int(value)
            except ValueError:
                pass
            try:
                return float(value)
            except ValueError:
                pass
        return value
    return _original_evaluate(expr, row)

evaluator.evaluate_expression = patched_evaluate_expression
logger.info("Applied monkey patch to fix literal type conversion")

def load_jsonl_file(filename, dynamic_where=None, dynamic_columns=None):
    """
    Generator function to read JSONL file line by line with column pruning support.
    
    Args:
        filename: Path to JSONL file
        dynamic_where: Optional WHERE clause (for filter pushdown - not implemented here)
        dynamic_columns: Optional list of column names to read (for column pruning)
    """
    with open(filename, 'r', encoding='utf-8') as f:
        for line in f:
            if not line.strip():
                continue
            
            row = json.loads(line)
            
            # ✅ COLUMN PRUNING: Only yield requested columns
            if dynamic_columns:
                # Filter to only requested columns
                pruned_row = {k: v for k, v in row.items() if k in dynamic_columns}
                yield pruned_row
            else:
                # No pruning requested, yield all columns
                yield row

def load_reference_data():
    categories = {cat['comp_category_id']: cat['name'] for cat in [json.loads(line) for line in open('categories.jsonl', 'r')]}
    
    shopflix_specs_data = [json.loads(line) for line in open('shopflix_specs.jsonl', 'r')]
    shopflix_specs = {spec['spec_id']: spec['spec_name'] for spec in shopflix_specs_data}
    shopflix_categories = {spec['category_id']: spec['category_name'] for spec in shopflix_specs_data}
    
    mapping_dict = {}
    for mapping in [json.loads(line) for line in open('specs_mapping.jsonl', 'r')]:
        key = (mapping['type'], mapping['shopflix_categoryid'])
        mapping_dict[key] = mapping['shopflix_specid']
    
    return categories, shopflix_specs, shopflix_categories, mapping_dict

def main():
    start_time = time.time()
    logger.info("Starting fast export with Streaming SQL Engine (POLARS + COLUMN PRUNING)")
    
    # Clear output file
    with open('fast_export.jsonl', 'w', encoding='utf-8') as f:
        pass
    
    # Load reference data (small, stays in memory)
    logger.info("Loading reference data...")
    categories, shopflix_specs, shopflix_categories, mapping_dict = load_reference_data()
    
    # Initialize SQL engine WITH POLARS + COLUMN PRUNING
    logger.info("Initializing Streaming SQL Engine with Polars + Column Pruning...")
    engine = Engine(
        debug=True,
        use_polars=True,       # ✅ Enable Polars for fast joins
        first_match_only=True  # Prevent cartesian products from duplicates
    )
    
    # Register tables with column pruning support
    logger.info("Registering tables with column pruning support...")
    
    # ✅ Sources now accept dynamic_columns parameter for column pruning
    engine.register(
        'products', 
        lambda dynamic_where=None, dynamic_columns=None: load_jsonl_file('products.jsonl', dynamic_where, dynamic_columns)
        # No filename - uses Polars Join (fast, but loads into memory)
        # With column pruning, memory usage is reduced by 50-90%
    )
    engine.register(
        'images', 
        lambda dynamic_where=None, dynamic_columns=None: load_jsonl_file('images.jsonl', dynamic_where, dynamic_columns)
    )
    engine.register(
        'offers', 
        lambda dynamic_where=None, dynamic_columns=None: load_jsonl_file('offers.jsonl', dynamic_where, dynamic_columns)
    )
    engine.register(
        'specs', 
        lambda dynamic_where=None, dynamic_columns=None: load_jsonl_file('specs.jsonl', dynamic_where, dynamic_columns)
    )
    
    logger.info("Tables registered with Polars Join + Column Pruning:")
    logger.info("  - Polars Join: Fast SIMD-accelerated joins")
    logger.info("  - Column Pruning: Only reads columns needed for query")
    logger.info("  - Memory: Reduced by 50-90% compared to reading all columns")
    logger.info("  - Speed: 10-50x faster than Python joins")
    
    # ONE streaming query with JOINs - processes everything in one pass!
    logger.info("Executing streaming query with joins...")
    
    # ✅ Query selects specific columns - column pruning will be applied automatically!
    query = """
        SELECT 
            products.product_id,
            products.product_sku,
            products.title,
            products.product_mpn,
            products.matching_sf_products,
            products.shopflix_category_id,
            products.main_category_id,
            products.brand,
            products.parent_id,
            images.image,
            offers.offer_title,
            offers.offer_seller,
            offers.offer_seller_id,
            offers.price,
            specs.type,
            specs.value
        FROM products
        LEFT JOIN images ON products.product_id = images.product_id
        LEFT JOIN offers ON products.product_id = offers.product_id
        LEFT JOIN specs ON products.product_id = specs.product_id
        WHERE products.checked = 1
    """
    
    # Time the query execution
    query_start = time.time()
    
    # Monitor memory if available
    if PSUTIL_AVAILABLE:
        process = psutil.Process()
        mem_before = process.memory_info().rss / 1024 / 1024  # MB
        logger.info(f"Memory before query: {mem_before:.1f} MB")
    
    # Stream results and aggregate by product (OPTIMIZED with Polars batch writing)
    logger.info("Streaming and aggregating results...")
    current_product = None
    current_data = None
    total_exported = 0
    
    # Use sets for O(1) duplicate checking (much faster than list membership)
    seen_images = set()
    seen_offers = set()
    seen_features = set()
    
    # Buffer for Polars batch writing (much faster than individual writes)
    products_buffer = []
    BUFFER_SIZE = 100000  # Write every 100,000 products using Polars
    
    # Use Polars for fast batch writing if available
    use_polars_write = POLARS_AVAILABLE
    
    if use_polars_write:
        logger.info("Using Polars for fast batch writing (vectorized JSONL export)")
    
    with open('fast_export.jsonl', 'wb') as output_file:
        for row in engine.query(query):
            pid = row['products.product_id']  # Note: columns are prefixed with table alias
            
            # New product - write previous and start new
            if current_product is None or pid != current_product:
                # Write previous product to buffer
                if current_data is not None:
                    products_buffer.append(current_data)
                    total_exported += 1
                    
                    # Flush buffer when full using Polars (much faster)
                    if len(products_buffer) >= BUFFER_SIZE:
                        if use_polars_write:
                            # Use Polars for fast batch writing (vectorized, SIMD-accelerated)
                            df = pl.DataFrame(products_buffer)
                            # Write to BytesIO buffer, then write to file
                            from io import BytesIO
                            buffer = BytesIO()
                            df.write_ndjson(buffer)  # NDJSON = JSONL format
                            output_file.write(buffer.getvalue())
                        else:
                            # Fallback: manual JSON serialization
                            json_lines = '\n'.join(json.dumps(p, ensure_ascii=False) for p in products_buffer)
                            output_file.write(json_lines.encode('utf-8') + b'\n')
                        products_buffer = []
                    
                    if total_exported % 10000 == 0:
                        logger.info(f"Exported {total_exported:,} products...")
                
                # Start new product
                current_product = pid
                seen_images.clear()
                seen_offers.clear()
                seen_features.clear()
                
                # Note: Column names are prefixed with table alias (e.g., 'products.product_id')
                current_data = {
                    'id': pid,
                    'product_sku': row.get('products.product_sku'),
                    'title': row.get('products.title'),
                    'product_mpn': row.get('products.product_mpn'),
                    'matching_sf_products': row.get('products.matching_sf_products'),
                    'category_id_shopflix': row.get('products.shopflix_category_id'),
                    'category_name_shopflix': shopflix_categories.get(row.get('products.shopflix_category_id'), ''),
                    'category_id_tgn': row.get('products.main_category_id'),
                    'category_name_tgn': categories.get(row.get('products.main_category_id'), ''),
                    'brand': row.get('products.brand'),
                    'parent_id': row.get('products.parent_id'),
                    'images': [],
                    'offers': [],
                    'features': []
                }
            
            # Accumulate data for current product (using sets for fast duplicate checking)
            image = row.get('images.image')
            if image:
                if image not in seen_images:  # O(1) lookup instead of O(n)
                    seen_images.add(image)
                    current_data['images'].append(image)
            
            offer_title = row.get('offers.offer_title')
            if offer_title:
                offer_key = (offer_title, row.get('offers.offer_seller'), row.get('offers.offer_seller_id'), row.get('offers.price'))
                if offer_key not in seen_offers:  # O(1) lookup
                    seen_offers.add(offer_key)
                    current_data['offers'].append({
                        'offer_title': offer_title,
                        'offer_seller': row.get('offers.offer_seller'),
                        'offer_seller_id': row.get('offers.offer_seller_id'),
                        'price': row.get('offers.price')
                    })
            
            spec_type = row.get('specs.type')
            if spec_type:
                feature_key = (spec_type, row.get('products.shopflix_category_id'), row.get('specs.value'))
                if feature_key not in seen_features:  # O(1) lookup
                    seen_features.add(feature_key)
                    key = (spec_type, row.get('products.shopflix_category_id'))
                    feature_id = mapping_dict.get(key)
                    feature_name = shopflix_specs.get(feature_id)
                    current_data['features'].append({
                        'feature_id': feature_id,
                        'feature_name': feature_name,
                        'feature_value': row.get('specs.value')
                    })
        
        # Write last product
        if current_data is not None:
            products_buffer.append(current_data)
            total_exported += 1
        
        # Flush remaining buffer
        if products_buffer:
            if use_polars_write:
                # Use Polars for fast batch writing (vectorized, SIMD-accelerated)
                df = pl.DataFrame(products_buffer)
                from io import BytesIO
                buffer = BytesIO()
                df.write_ndjson(buffer)  # NDJSON = JSONL format
                output_file.write(buffer.getvalue())
            else:
                # Fallback: manual JSON serialization
                json_lines = '\n'.join(json.dumps(p, ensure_ascii=False) for p in products_buffer)
                output_file.write(json_lines.encode('utf-8') + b'\n')
    
    query_time = time.time() - query_start
    total_time = time.time() - start_time
    
    # Memory after query
    if PSUTIL_AVAILABLE:
        mem_after = process.memory_info().rss / 1024 / 1024  # MB
        mem_used = mem_after - mem_before
        logger.info("=" * 60)
        logger.info(f"OPTIMIZATION RESULTS:")
        logger.info(f"  Query execution time: {query_time:.2f} seconds")
        logger.info(f"  Total time: {total_time:.2f} seconds")
        logger.info(f"  Products exported: {total_exported:,}")
        logger.info(f"  Rows/second: {total_exported / query_time:,.0f}")
        logger.info(f"  Memory used: {mem_used:.1f} MB (peak: {mem_after:.1f} MB)")
        logger.info("=" * 60)
    else:
        logger.info("=" * 60)
        logger.info(f"OPTIMIZATION RESULTS:")
        logger.info(f"  Query execution time: {query_time:.2f} seconds")
        logger.info(f"  Total time: {total_time:.2f} seconds")
        logger.info(f"  Products exported: {total_exported:,}")
        logger.info(f"  Rows/second: {total_exported / query_time:,.0f}")
        logger.info("=" * 60)
    
    logger.info("Check debug output above to see:")
    logger.info("  ✅ Column pruning: Engine will pass only required columns to sources")
    logger.info("  ✅ Polars Join: Fast SIMD-accelerated joins")
    logger.info("  ✅ Memory reduction: 50-90% less memory due to column pruning")
    logger.info("  ✅ Speed: 10-50x faster than Python joins")

if __name__ == "__main__":
    main()











