"""
Main Engine class - public API for the streaming SQL engine.
"""

from typing import Any
from .parser import parse_sql
from .planner import build_logical_plan
from .executor import execute_plan
from .jsonl_executor import execute_plan_jsonl


class Engine:
    """
    Main interface for the streaming SQL execution engine.
    
    Usage:
        engine = Engine(debug=True)  # Enable debug mode
        engine.register("users", lambda: iter([{"id": 1, "name": "Alice"}]))
        results = engine.query("SELECT users.name FROM users WHERE users.id = 1")
        for row in results:
            print(row)
    """
    
    def __init__(self, debug=False, use_jsonl_mode=False, use_polars=True):
        """
        Initialize a new engine instance.
        
        Args:
            debug: If True, enables verbose logging of execution stages
            use_jsonl_mode: If True, uses JSONL-based execution (export tables to JSONL, then merge)
                           This is simpler and can reduce CPU usage for complex queries
            use_polars: If True, uses Polars for optimizations when available (default: True)
        """
        self._sources = {}
        self._source_metadata = {}
        self.debug = debug
        self.use_jsonl_mode = use_jsonl_mode
        self.use_polars = use_polars
    
    def register(
        self,
        table_name,
        source_fn,
        ordered_by=None,
        is_database_source=False,
        filename=None
    ):
        """
        Register a table source.
        
        Args:
            table_name: Name of the table as used in SQL queries
            source_fn: Function that returns an iterator of row dictionaries
            ordered_by: Optional column name if the source is sorted by this column
                       (enables merge joins)
            is_database_source: If True, indicates this is a database source that supports
                               column pruning and filter pushdown optimizations
            filename: Optional filename if source is file-based (enables mmap-based joins
                     for 90-99% memory reduction)
        """
        if not callable(source_fn):
            raise ValueError(f"source_fn must be callable, got {type(source_fn)}")
        
        self._sources[table_name] = source_fn
        self._source_metadata[table_name] = {
            "ordered_by": ordered_by,
            "is_database_source": is_database_source,
            "filename": filename  # Enable mmap-based joins
        }
    
    def query(self, sql):
        """
        Execute a SQL query and return a generator of result rows.
        
        Args:
            sql: SQL query string
            
        Returns:
            Generator yielding dictionaries representing result rows
            
        Raises:
            ValueError: If query contains unsupported constructs
            KeyError: If referenced table is not registered
        """
        if self.debug:
            print("=" * 60)
            print("STREAMING SQL ENGINE - DEBUG MODE")
            print("=" * 60)
            print(f"\n[1/3] PARSING SQL QUERY...")
            print(f"Query:\n{sql}\n")
        
        # Parse SQL into AST
        ast = parse_sql(sql)
        
        if self.debug:
            print(f"✓ SQL parsed successfully")
            print(f"\n[2/3] BUILDING LOGICAL PLAN...")
        
        # Build logical plan
        logical_plan = build_logical_plan(ast, self._sources.keys())
        
        if self.debug:
            print(f"✓ Logical plan built:")
            print(f"  - Root table: {logical_plan.root_table} (alias: {logical_plan.root_alias})")
            print(f"  - Joins: {len(logical_plan.joins)}")
            for i, join in enumerate[Any](logical_plan.joins, 1):
                print(f"    {i}. {join.join_type} JOIN {join.table} ON {join.left_key} = {join.right_key}")
            print(f"  - WHERE clause: {'Yes' if logical_plan.where_expr else 'No'}")
            print(f"  - Projections: {len(logical_plan.projections)}")
            print(f"\n[3/3] EXECUTING QUERY...")
            print(f"Building execution pipeline...\n")
        
        # Execute plan
        if self.use_jsonl_mode:
            if self.debug:
                print(f"\nUsing JSONL-based execution mode (simpler, lower CPU)\n")
            return execute_plan_jsonl(logical_plan, self._sources, self._source_metadata, debug=self.debug)
        else:
            return execute_plan(logical_plan, self._sources, self._source_metadata, debug=self.debug, use_polars=self.use_polars)
