"""
Database connector utilities for PostgreSQL and MySQL.
Provides connection pooling and streaming row iterators.
"""

import os
from contextlib import contextmanager

# PostgreSQL imports
try:
    import psycopg2
    from psycopg2 import pool
    from psycopg2.extras import RealDictCursor as PgRealDictCursor
    PSYCOPG2_AVAILABLE = True
except ImportError:
    PSYCOPG2_AVAILABLE = False
    
# MySQL imports
try:
    import pymysql
    from dbutils.pooled_db import PooledDB
    PYMYSQL_AVAILABLE = True
except ImportError:
    PYMYSQL_AVAILABLE = False

# MongoDB imports
try:
    from pymongo import MongoClient
    from pymongo.collection import Collection
    PYMONGO_AVAILABLE = True
except ImportError:
    PYMONGO_AVAILABLE = False
    MongoClient = None
    Collection = None
    

class PostgreSQLPool:
    """
    PostgreSQL connection pool manager.
    """
    
    def __init__(
        self,
        host,
        port,
        user,
        password,
        database,
        minconn=1,
        maxconn=200
    ):
        """
        Initialize PostgreSQL connection pool.
        
        Args:
            host: Database host
            port: Database port
            user: Database user
            password: Database password
            database: Database name
            minconn: Minimum number of connections
            maxconn: Maximum number of connections
        """
        self.pool = psycopg2.pool.ThreadedConnectionPool(
            minconn=minconn,
            maxconn=maxconn,
            host=host,
            port=port,
            user=user,
            password=password,
            database=database
        )
    
    @contextmanager
    def get_connection(self):
        """Get a connection from the pool (context manager)."""
        conn = self.pool.getconn()
        try:
            yield conn
        finally:
            self.pool.putconn(conn)
    
    def closeall(self):
        """Close all connections in the pool."""
        self.pool.closeall()


class MySQLPool:
    """
    MySQL connection pool manager using pymysql and PooledDB.
    """
    
    def __init__(
        self,
        host,
        port,
        user,
        password,
        database,
        maxconnections=200
    ):
        """
        Initialize MySQL connection pool.
        
        Args:
            host: Database host
            port: Database port
            user: Database user
            password: Database password
            database: Database name
            maxconnections: Maximum number of connections
        """
        if not PYMYSQL_AVAILABLE:
            raise ImportError(
                "pymysql and dbutils are required for MySQL support. "
                "Install with: pip install pymysql dbutils"
            )
        
        self.pool = PooledDB(
            creator=pymysql,
            maxconnections=maxconnections,
            host=host,
            port=port,
            user=user,
            password=password,
            database=database,
            cursorclass=pymysql.cursors.DictCursor
        )
    
    @contextmanager
    def get_connection(self):
        """Get a connection from the pool (context manager)."""
        conn = self.pool.connection()
        try:
            yield conn
        finally:
            conn.close()
    
    def closeall(self):
        """Close all connections in the pool."""
        self.pool.close()


class MongoPool:
    """
    MongoDB connection manager.
    MongoDB doesn't use traditional connection pooling like SQL databases,
    but we provide a similar interface for consistency.
    """
    
    def __init__(
        self,
        host='localhost',
        port=27017,
        username=None,
        password=None,
        database=None,
        auth_source='admin',
        max_pool_size=100
    ):
        """
        Initialize MongoDB connection.
        
        Args:
            host: MongoDB host (default: localhost)
            port: MongoDB port (default: 27017)
            username: MongoDB username (optional)
            password: MongoDB password (optional)
            database: Database name (optional, can be specified per collection)
            auth_source: Authentication database (default: admin)
            max_pool_size: Maximum pool size for connections (default: 100)
        """
        if not PYMONGO_AVAILABLE:
            raise ImportError(
                "pymongo is required for MongoDB support. "
                "Install with: pip install pymongo"
            )
        
        # Build connection URI
        if username and password:
            uri = f"mongodb://{username}:{password}@{host}:{port}/{database or ''}?authSource={auth_source}"
        else:
            uri = f"mongodb://{host}:{port}/{database or ''}"
        
        self.client = MongoClient(uri, maxPoolSize=max_pool_size)
        self.database_name = database
        self._database = None
        
        # Test connection
        try:
            self.client.admin.command('ping')
        except Exception as e:
            raise ConnectionError(f"Failed to connect to MongoDB: {e}")
    
    def get_database(self, database_name=None):
        """Get a database instance."""
        db_name = database_name or self.database_name
        if not db_name:
            raise ValueError("Database name must be provided")
        return self.client[db_name]
    
    @contextmanager
    def get_connection(self):
        """Get MongoDB client (context manager for consistency with SQL pools)."""
        # MongoDB client is thread-safe and handles pooling internally
        yield self.client
    
    def closeall(self):
        """Close MongoDB connection."""
        self.client.close()


def create_pool_from_env(db_type=None):
    """
    Create a database pool from environment variables.
    Auto-detects database type based on port if not specified.
    
    Expected environment variables:
    - db_host
    - db_port
    - db_user
    - db_password
    - db_name
    
    Args:
        db_type: Optional database type ('postgresql' or 'mysql'). 
                 If None, auto-detects based on port (3306=MySQL, 5432=PostgreSQL)
    
    Returns:
        PostgreSQLPool or MySQLPool instance
        
    Raises:
        ValueError: If port/db_type mismatch or unsupported database type
        ImportError: If required database driver is not installed
    """
    host = os.environ.get('db_host', 'localhost')
    port = int(os.environ.get('db_port', 5432))
    user = os.environ.get('db_user', 'postgres')
    password = os.environ.get('db_password', '')
    database = os.environ.get('db_name', 'postgres')
    
    # Auto-detect database type if not specified
    if db_type is None:
        if port == 3306:
            db_type = 'mysql'
        elif port == 5432:
            db_type = 'postgresql'
        else:
            # Default to PostgreSQL for unknown ports, but warn
            import warnings
            warnings.warn(
                f"Port {port} is not a standard database port. "
                "Defaulting to PostgreSQL. If using MySQL, set db_type='mysql' or use port 3306.",
                UserWarning
            )
            db_type = 'postgresql'
    
    # Validate port/db_type combination
    if db_type == 'mysql' and port != 3306:
        import warnings
        warnings.warn(
            f"MySQL typically uses port 3306, but port {port} is specified. "
            "Continuing anyway...",
            UserWarning
        )
    elif db_type == 'postgresql' and port == 3306:
        raise ValueError(
            f"Port 3306 is typically MySQL, but db_type='postgresql' was specified. "
            "If you're using MySQL, set db_type='mysql' or use create_mysql_pool_from_env()."
        )
    
    if db_type == 'mysql':
        if not PYMYSQL_AVAILABLE:
            raise ImportError(
                "pymysql and dbutils are required for MySQL support. "
                "Install with: pip install pymysql dbutils"
            )
        return MySQLPool(host, port, user, password, database)
    elif db_type == 'postgresql':
        if not PSYCOPG2_AVAILABLE:
            raise ImportError(
                "psycopg2 is required for PostgreSQL support. "
                "Install with: pip install psycopg2-binary"
            )
        return PostgreSQLPool(host, port, user, password, database)
    else:
        raise ValueError(f"Unsupported database type: {db_type}. Use 'postgresql' or 'mysql'.")


def create_mysql_pool_from_env():
    """
    Create a MySQL pool from environment variables.
    Convenience function for MySQL connections.
    
    Expected environment variables:
    - db_host
    - db_port (defaults to 3306 if not set)
    - db_user
    - db_password
    - db_name
    """
    return create_pool_from_env(db_type='mysql')


def create_postgresql_pool_from_env():
    """
    Create a PostgreSQL pool from environment variables.
    Convenience function for PostgreSQL connections.
    
    Expected environment variables:
    - db_host
    - db_port (defaults to 5432 if not set)
    - db_user
    - db_password
    - db_name
    """
    return create_pool_from_env(db_type='postgresql')


def create_mongo_pool_from_env():
    """
    Create a MongoDB connection from environment variables.
    
    Expected environment variables:
    - mongo_host (defaults to localhost)
    - mongo_port (defaults to 27017)
    - mongo_user (optional)
    - mongo_password (optional)
    - mongo_database (optional, can be specified per collection)
    - mongo_auth_source (defaults to admin)
    
    Returns:
        MongoPool instance
    """
    host = os.environ.get('mongo_host', 'localhost')
    port = int(os.environ.get('mongo_port', 27017))
    username = os.environ.get('mongo_user')
    password = os.environ.get('mongo_password')
    database = os.environ.get('mongo_database')
    auth_source = os.environ.get('mongo_auth_source', 'admin')
    
    return MongoPool(
        host=host,
        port=port,
        username=username,
        password=password,
        database=database,
        auth_source=auth_source
    )


def create_mongo_pool_from_env():
    """
    Create a MongoDB connection from environment variables.
    
    Expected environment variables:
    - mongo_host (defaults to localhost)
    - mongo_port (defaults to 27017)
    - mongo_user (optional)
    - mongo_password (optional)
    - mongo_database (optional, can be specified per collection)
    - mongo_auth_source (defaults to admin)
    
    Returns:
        MongoPool instance
    """
    host = os.environ.get('mongo_host', 'localhost')
    port = int(os.environ.get('mongo_port', 27017))
    username = os.environ.get('mongo_user')
    password = os.environ.get('mongo_password')
    database = os.environ.get('mongo_database')
    auth_source = os.environ.get('mongo_auth_source', 'admin')
    
    return MongoPool(
        host=host,
        port=port,
        username=username,
        password=password,
        database=database,
        auth_source=auth_source
    )


def stream_query(
    pool,
    query,
    params=None,
    fetch_size=1000
):
    """
    Stream rows from a database query as a generator.
    Supports both PostgreSQL and MySQL.
    
    Args:
        pool: Database connection pool (PostgreSQLPool or MySQLPool)
        query: SQL query string
        params: Optional query parameters (for parameterized queries)
        fetch_size: Number of rows to fetch at a time
        
    Yields:
        Dictionary representing each row
        
    Example:
        pool = create_pool_from_env()
        for row in stream_query(pool, "SELECT * FROM users WHERE active = %s", (True,)):
            print(row)
    """
    with pool.get_connection() as conn:
        if isinstance(pool, PostgreSQLPool):
            # PostgreSQL: Use server-side cursor for efficient streaming
            cursor = conn.cursor(
                name='streaming_cursor',
                cursor_factory=PgRealDictCursor
            )
            
            try:
                # Execute query with server-side cursor
                if params:
                    cursor.execute(query, params)
                else:
                    cursor.execute(query)
                
                # Stream rows in batches
                while True:
                    rows = cursor.fetchmany(fetch_size)
                    if not rows:
                        break
                    
                    for row in rows:
                        # Convert RealDictRow to regular dict
                        yield dict(row)
            finally:
                cursor.close()
        
        elif isinstance(pool, MySQLPool):
            # MySQL: Use regular cursor with fetchmany
            cursor = conn.cursor()
            
            try:
                # Execute query
                if params:
                    cursor.execute(query, params)
                else:
                    cursor.execute(query)
                
                # Stream rows in batches
                while True:
                    rows = cursor.fetchmany(fetch_size)
                    if not rows:
                        break
                    
                    for row in rows:
                        # pymysql DictCursor already returns dicts
                        yield dict(row)
            finally:
                cursor.close()
        
        elif isinstance(pool, MongoPool):
            # MongoDB: Collections are accessed differently
            # This function is mainly for SQL databases
            # For MongoDB, use create_mongo_collection_source instead
            raise ValueError(
                "stream_query is for SQL databases. "
                "For MongoDB, use create_mongo_collection_source() instead."
            )
        else:
            raise ValueError(f"Unsupported pool type: {type(pool)}")


def create_table_source(
    pool,
    table_name,
    where_clause=None,
    order_by=None,
    columns=None,
    _dynamic_where=None,
    _dynamic_columns=None
):
    """
    Create a source function for a database table.
    Supports both PostgreSQL and MySQL.
    
    Args:
        pool: Database connection pool (PostgreSQLPool or MySQLPool)
        table_name: Name of the table
        where_clause: Optional WHERE clause (without WHERE keyword)
        order_by: Optional ORDER BY clause (without ORDER BY keyword)
        columns: Optional list/set of column names to select (None = all columns)
                 This enables column pruning for better performance
        _dynamic_where: Internal - allows WHERE to be set dynamically at query time
        _dynamic_columns: Internal - allows columns to be set dynamically at query time
        
    Returns:
        A function that returns an iterator of row dictionaries
        
    Example:
        pool = create_pool_from_env()
        source_fn = create_table_source(pool, "users", "active = 1", "id")
        engine.register("users", source_fn, ordered_by="id", is_database_source=True)
        
        # With column pruning:
        source_fn = create_table_source(pool, "users", columns=["id", "name", "email"])
        engine.register("users", source_fn, is_database_source=True)
    """
    def source_fn(dynamic_where=None, dynamic_columns=None):
        # Use dynamic parameters if provided (for query-time optimization)
        # Otherwise use static parameters
        final_where = dynamic_where if dynamic_where is not None else where_clause
        final_columns = dynamic_columns if dynamic_columns is not None else columns
        
        # Build column list
        if final_columns:
            # Column pruning: only select needed columns
            column_list = ", ".join(final_columns)
            query = f"SELECT {column_list} FROM {table_name}"
        else:
            query = f"SELECT * FROM {table_name}"
        
        # Push WHERE clause to database (filter pushdown)
        if final_where:
            query += f" WHERE {final_where}"
        if order_by:
            query += f" ORDER BY {order_by}"
        
        return stream_query(pool, query)
    
    return source_fn


def create_mongo_collection_source(
    pool,
    collection_name,
    database_name=None,
    filter_query=None,
    projection=None,
    sort=None,
    limit=None
):
    """
    Create a source function for a MongoDB collection.
    
    Args:
        pool: MongoPool instance
        collection_name: Name of the collection
        database_name: Database name (uses pool's default if not provided)
        filter_query: Optional MongoDB filter query (dict) - equivalent to WHERE clause
        projection: Optional projection dict - equivalent to SELECT columns
                    Example: {"name": 1, "email": 1, "_id": 0}
        sort: Optional sort specification (list of tuples) - equivalent to ORDER BY
              Example: [("created_at", -1)] for descending
        limit: Optional limit on number of documents
        
    Returns:
        A function that returns an iterator of row dictionaries
        
    Example:
        pool = create_mongo_pool_from_env()
        source_fn = create_mongo_collection_source(
            pool, 
            "users",
            filter_query={"active": True},
            projection={"name": 1, "email": 1, "_id": 0},
            sort=[("created_at", -1)]
        )
        engine.register("users", source_fn)
    """
    def source_fn():
        db = pool.get_database(database_name)
        collection = db[collection_name]
        
        # Build query
        cursor = collection.find(filter_query or {})
        
        # Apply projection if specified
        if projection:
            cursor = cursor.projection(projection)
        
        # Apply sort if specified
        if sort:
            cursor = cursor.sort(sort)
        
        # Apply limit if specified
        if limit:
            cursor = cursor.limit(limit)
        
        # Stream documents
        for doc in cursor:
            # Convert ObjectId to string for JSON serialization
            if '_id' in doc:
                doc['_id'] = str(doc['_id'])
            yield doc
    
    return source_fn

