"""
MCP server implementation for Swagger/OpenAPI tools.
"""

import asyncio
import json
import sys
import traceback
from typing import Any, Dict, List, Optional

from .services.loader import SwaggerLoader
from .tools import TOOL_DEFINITIONS, TOOL_HANDLERS


class SwaggerMCPServer:
    """MCP server for Swagger/OpenAPI tools."""

    def __init__(self):
        self.loader = SwaggerLoader()
        self.initialized = False

    async def read_message(self) -> Optional[Dict[str, Any]]:
        """Read a JSON-RPC message from stdin."""
        try:
            line = await asyncio.get_event_loop().run_in_executor(
                None, sys.stdin.readline
            )
            if not line:
                return None
            return json.loads(line.strip())
        except (json.JSONDecodeError, EOFError):
            return None

    async def write_message(self, message: Dict[str, Any]) -> None:
        """Write a JSON-RPC message to stdout."""
        json_str = json.dumps(message)
        await asyncio.get_event_loop().run_in_executor(
            None, lambda: sys.stdout.write(json_str + "\n")
        )
        sys.stdout.flush()

    async def handle_request(self, request: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Handle a JSON-RPC request."""
        method = request.get("method")
        request_id = request.get("id")

        if not method or not request_id:
            return None

        try:
            if method == "initialize":
                return await self.handle_initialize(request, request_id)
            elif method == "tools/list":
                return await self.handle_tools_list(request, request_id)
            elif method == "tools/call":
                return await self.handle_tools_call(request, request_id)
            else:
                return self.create_error_response(
                    request_id, -32601, "Method not found", data={"method": method}
                )
        except Exception as e:
            traceback.print_exc()
            return self.create_error_response(
                request_id, -32603, "Internal error", data={"error": str(e)}
            )

    async def handle_initialize(self, request: Dict[str, Any], request_id: Any) -> Dict[str, Any]:
        """Handle initialize request."""
        self.initialized = True
        return {
            "jsonrpc": "2.0",
            "id": request_id,
            "result": {
                "protocolVersion": "2024-11-05",
                "capabilities": {
                    "tools": {}
                },
                "serverInfo": {
                    "name": "swagger-mcp-python",
                    "version": "1.0.0"
                }
            }
        }

    async def handle_tools_list(self, request: Dict[str, Any], request_id: Any) -> Dict[str, Any]:
        """Handle tools/list request."""
        return {
            "jsonrpc": "2.0",
            "id": request_id,
            "result": {
                "tools": TOOL_DEFINITIONS
            }
        }

    async def handle_tools_call(self, request: Dict[str, Any], request_id: Any) -> Dict[str, Any]:
        """Handle tools/call request."""
        params = request.get("params", {})
        tool_name = params.get("name")
        arguments = params.get("arguments", {})

        if not tool_name:
            return self.create_error_response(
                request_id, -32602, "Invalid params", data={"message": "Missing tool name"}
            )

        if tool_name not in TOOL_HANDLERS:
            return self.create_error_response(
                request_id, -32601, "Method not found", data={"message": f"Tool '{tool_name}' not found"}
            )

        try:
            handler = TOOL_HANDLERS[tool_name]
            result = handler(self.loader, arguments)
            return {
                "jsonrpc": "2.0",
                "id": request_id,
                "result": {
                    "content": result.get("content", []),
                    "isError": result.get("isError", False)
                }
            }
        except Exception as e:
            return self.create_error_response(
                request_id, -32603, "Internal error", data={"error": str(e)}
            )

    def create_error_response(self, request_id: Any, code: int, message: str,
                             data: Optional[Dict[str, Any]] = None) -> Dict[str, Any]:
        """Create a JSON-RPC error response."""
        error = {
            "code": code,
            "message": message
        }
        if data:
            error["data"] = data

        return {
            "jsonrpc": "2.0",
            "id": request_id,
            "error": error
        }

    async def run(self) -> None:
        """Run the MCP server."""
        print("Swagger MCP Server (Python) starting...", file=sys.stderr)

        while True:
            try:
                message = await self.read_message()
                if message is None:
                    break

                # Check if it's a request (has method) or response
                if "method" in message:
                    response = await self.handle_request(message)
                    if response:
                        await self.write_message(response)
                # Ignore responses for now

            except KeyboardInterrupt:
                break
            except Exception as e:
                print(f"Error in main loop: {e}", file=sys.stderr)
                traceback.print_exc()

        print("Swagger MCP Server (Python) shutting down...", file=sys.stderr)