"""
Swagger document loader with caching and authentication.
"""

import sys
import time
from typing import Dict, Optional, Tuple, Any
from urllib.parse import urljoin, urlparse

import requests
from pydantic import ValidationError

from ..types.swagger import SwaggerDoc


class SwaggerLoader:
    """Load and cache Swagger documents with authentication support."""

    def __init__(self):
        self.services: Dict[str, str] = {}
        self.cache: Dict[str, SwaggerDoc] = {}
        self.cache_timestamp: Dict[str, float] = {}
        self.CACHE_TTL = 5 * 60  # 5 minutes in seconds

        self.auth_token: Optional[str] = None
        self.credentials: Optional[Dict[str, str]] = None

        self._parse_args()

    def _parse_args(self) -> None:
        """Parse command line arguments."""
        args = sys.argv[1:]
        if not args:
            self.services["default"] = "http://localhost:8090/v3/api-docs"
            print("No arguments provided. Using default: http://localhost:8090/v3/api-docs",
                  file=sys.stderr)
            return

        for arg in args:
            if arg.startswith("auth="):
                # Format: auth=user:pass OR auth=/login/path:user:pass
                val = arg[5:]
                parts = val.split(":")
                if len(parts) == 2:
                    self.credentials = {"user": parts[0], "pass": parts[1]}
                    print(f"[Auth] Credentials loaded for user: {parts[0]}", file=sys.stderr)
                elif len(parts) == 3:
                    self.credentials = {
                        "loginPath": parts[0],
                        "user": parts[1],
                        "pass": parts[2]
                    }
                    print(f"[Auth] Auto-login configured. Path: {parts[0]}, User: {parts[1]}",
                          file=sys.stderr)
            elif "=" in arg:
                name, url = arg.split("=", 1)
                self.services[name] = url
                print(f"Registered service '{name}': {url}", file=sys.stderr)
            else:
                if arg.startswith("http"):
                    self.services["default"] = arg
                    print(f"Registered default service: {arg}", file=sys.stderr)
                else:
                    print(f"Ignoring invalid argument: {arg}. Expected format: name=url or http://...",
                          file=sys.stderr)

    def set_auth_token(self, token: str) -> None:
        """Set the authentication token."""
        self.auth_token = token
        print("[Auth] Token cached successfully.", file=sys.stderr)

    def get_auth_token(self) -> Optional[str]:
        """Get the authentication token."""
        return self.auth_token

    def get_credentials(self) -> Optional[Dict[str, str]]:
        """Get stored credentials."""
        return self.credentials

    def get_services(self) -> Dict[str, str]:
        """Get all configured services."""
        return self.services

    def get_doc(self, service_name: Optional[str] = None, force_refresh: bool = False) -> Tuple[SwaggerDoc, str, str]:
        """
        Get Swagger document for a service.

        Args:
            service_name: Name of the service, or None for single service
            force_refresh: Force refresh from source

        Returns:
            Tuple of (document, service_name, base_url)
        """
        name = service_name
        if not name:
            if len(self.services) == 1:
                name = list(self.services.keys())[0]
            else:
                raise ValueError(
                    f"Multiple services configured. Please specify 'service_name'. "
                    f"Available: {', '.join(self.services.keys())}"
                )

        if not name or name not in self.services:
            raise ValueError(
                f"Service '{name}' not found. Available: {', '.join(self.services.keys())}"
            )

        doc_url = self.services[name]
        doc: SwaggerDoc

        # Check cache with TTL
        now = time.time()
        timestamp = self.cache_timestamp.get(name, 0)
        is_expired = (now - timestamp) > self.CACHE_TTL

        if name in self.cache and not force_refresh and not is_expired:
            doc = self.cache[name]
        else:
            try:
                if is_expired and name in self.cache:
                    print(f"[Cache] Doc for '{name}' expired. Refreshing...", file=sys.stderr)
                else:
                    print(f"Fetching Swagger docs for '{name}' from {doc_url}...", file=sys.stderr)

                response = requests.get(doc_url, timeout=30)
                response.raise_for_status()
                doc_data = response.json()
                doc = SwaggerDoc(**doc_data)
                self.cache[name] = doc
                self.cache_timestamp[name] = now

            except requests.RequestException as error:
                # Retry with 127.0.0.1 if localhost failed
                if "localhost" in doc_url and (
                    isinstance(error, requests.ConnectionError) or
                    (hasattr(error, 'response') and error.response is None)
                ):
                    try:
                        ipv4_url = doc_url.replace("localhost", "127.0.0.1")
                        print(f"Retrying with 127.0.0.1: {ipv4_url}...", file=sys.stderr)
                        response = requests.get(ipv4_url, timeout=30)
                        response.raise_for_status()
                        doc_data = response.json()
                        doc = SwaggerDoc(**doc_data)
                        self.cache[name] = doc
                        self.cache_timestamp[name] = now
                        # Update service URL to avoid future errors
                        self.services[name] = ipv4_url
                    except requests.RequestException as retry_error:
                        msg = str(retry_error)
                        raise RuntimeError(
                            f"Failed to fetch Swagger docs for '{name}': {msg}. "
                            f"Please ensure the service at {doc_url} is running."
                        ) from retry_error
                else:
                    msg = str(error)
                    raise RuntimeError(
                        f"Failed to fetch Swagger docs for '{name}': {msg}. "
                        f"Please ensure the service at {doc_url} is running."
                    ) from error
            except ValidationError as error:
                raise RuntimeError(
                    f"Invalid Swagger document format for '{name}': {error}"
                ) from error

        # Calculate base URL
        base_url = ""
        if doc.servers and len(doc.servers) > 0:
            base_url = doc.servers[0].url
            if not base_url.startswith(("http://", "https://")):
                doc_url_parsed = urlparse(doc_url)
                base_url = urljoin(f"{doc_url_parsed.scheme}://{doc_url_parsed.netloc}", base_url)
        else:
            doc_url_parsed = urlparse(doc_url)
            base_url = f"{doc_url_parsed.scheme}://{doc_url_parsed.netloc}"

        if base_url.endswith("/"):
            base_url = base_url[:-1]

        return doc, name, base_url