"""
MCP tool implementations for Swagger/OpenAPI interaction.
"""

import json
import sys
from typing import Any, Dict, List, Optional
from urllib.parse import urljoin, urlencode

import requests

from ..services.loader import SwaggerLoader
from ..utils.schema import resolve_schema


# Tool definitions
TOOL_DEFINITIONS = [
    {
        "name": "list_services",
        "description": "List all configured Swagger services.",
        "inputSchema": {
            "type": "object",
            "properties": {}
        }
    },
    {
        "name": "refresh_docs",
        "description": "Force refresh the cached Swagger documentation from the source URL.",
        "inputSchema": {
            "type": "object",
            "properties": {
                "service_name": {
                    "type": "string",
                    "description": "Name of the service to refresh"
                }
            }
        }
    },
    {
        "name": "list_endpoints",
        "description": "List all available API endpoints.",
        "inputSchema": {
            "type": "object",
            "properties": {
                "service_name": {
                    "type": "string",
                    "description": "Name of the service"
                }
            }
        }
    },
    {
        "name": "search_apis",
        "description": "Search for APIs by keyword.",
        "inputSchema": {
            "type": "object",
            "properties": {
                "query": {
                    "type": "string",
                    "description": "The search keyword"
                },
                "service_name": {
                    "type": "string",
                    "description": "Name of the service to search in"
                }
            },
            "required": ["query"]
        }
    },
    {
        "name": "get_endpoint_details",
        "description": "Get full details of a specific API endpoint.",
        "inputSchema": {
            "type": "object",
            "properties": {
                "path": {"type": "string"},
                "method": {"type": "string"},
                "service_name": {"type": "string"}
            },
            "required": ["path", "method"]
        }
    },
    {
        "name": "debug_endpoint",
        "description": "Execute a real HTTP request to the API.",
        "inputSchema": {
            "type": "object",
            "properties": {
                "service_name": {"type": "string"},
                "path": {"type": "string"},
                "method": {"type": "string"},
                "path_params": {"type": "object"},
                "query_params": {"type": "object"},
                "headers": {"type": "object"},
                "body": {"type": "object"}
            },
            "required": ["path", "method"]
        }
    },
    {
        "name": "generate_curl",
        "description": "Generate a cURL command string.",
        "inputSchema": {
            "type": "object",
            "properties": {
                "service_name": {"type": "string"},
                "path": {"type": "string"},
                "method": {"type": "string"},
                "path_params": {"type": "object"},
                "query_params": {"type": "object"},
                "headers": {"type": "object"},
                "body": {"type": "object"}
            },
            "required": ["path", "method"]
        }
    }
]


def handle_list_services(loader: SwaggerLoader, args: Dict[str, Any]) -> Dict[str, Any]:
    """Handle list_services tool."""
    services = loader.get_services()
    services_list = [f"{name}: {url}" for name, url in services.items()]
    return {
        "content": [{
            "type": "text",
            "text": "\n".join(services_list)
        }]
    }


def handle_refresh_docs(loader: SwaggerLoader, args: Dict[str, Any]) -> Dict[str, Any]:
    """Handle refresh_docs tool."""
    service_name = args.get("service_name")
    try:
        doc, name, _ = loader.get_doc(service_name, force_refresh=True)
        title = doc.info.title if doc.info else "Unknown"
        return {
            "content": [{
                "type": "text",
                "text": f"Successfully refreshed documentation for '{name}'. Title: {title}"
            }]
        }
    except Exception as e:
        return {
            "content": [{
                "type": "text",
                "text": f"Error refreshing documentation: {str(e)}"
            }],
            "isError": True
        }


def handle_list_endpoints(loader: SwaggerLoader, args: Dict[str, Any]) -> Dict[str, Any]:
    """Handle list_endpoints tool."""
    service_name = args.get("service_name")
    try:
        doc, name, _ = loader.get_doc(service_name)
        endpoints = []
        for path, methods in doc.paths.items():
            for method, details in methods.items():
                summary = details.get("summary", "No summary")
                description = details.get("description", "")
                if description:
                    description = description[:100] + "..." if len(description) > 100 else description
                endpoints.append({
                    "service": name,
                    "method": method.upper(),
                    "path": path,
                    "summary": summary,
                    "description": description or None
                })
        return {
            "content": [{
                "type": "text",
                "text": json.dumps(endpoints, indent=2)
            }]
        }
    except Exception as e:
        return {
            "content": [{
                "type": "text",
                "text": f"Error listing endpoints: {str(e)}"
            }],
            "isError": True
        }


def handle_search_apis(loader: SwaggerLoader, args: Dict[str, Any]) -> Dict[str, Any]:
    """Handle search_apis tool."""
    query = str(args.get("query", "")).lower().strip()
    service_name = args.get("service_name")

    if not query:
        return {
            "content": [{
                "type": "text",
                "text": "[]"
            }]
        }

    terms = [t for t in query.split() if t]
    matches = []

    services_to_search = [service_name] if service_name else list(loader.get_services().keys())

    for s_name in services_to_search:
        try:
            doc, _, _ = loader.get_doc(s_name)
            for path, methods in doc.paths.items():
                for method, details in methods.items():
                    summary = details.get("summary", "").lower()
                    description = details.get("description", "").lower()
                    path_lower = path.lower()

                    score = 0
                    matched_term_count = 0

                    for term in terms:
                        term_score = 0
                        if term in path_lower:
                            term_score += 10  # Path match is most important
                        if term in summary:
                            term_score += 5   # Summary match is important
                        if term in description:
                            term_score += 1   # Description match is bonus

                        if term_score > 0:
                            score += term_score
                            matched_term_count += 1

                    # Only include if at least one term matched
                    if score > 0:
                        score += matched_term_count * 20  # Bonus for multiple term matches
                        matches.append({
                            "service": s_name,
                            "method": method.upper(),
                            "path": path,
                            "summary": details.get("summary", "No summary"),
                            "description": (details.get("description", "")[:200] + "...")
                                         if details.get("description") and len(details.get("description", "")) > 200
                                         else details.get("description"),
                            "score": score
                        })
        except Exception as e:
            print(f"Skipping search for {s_name}: {e}", file=sys.stderr)
            continue

    # Sort by score descending
    matches.sort(key=lambda x: x["score"], reverse=True)

    # Limit results
    top_matches = matches[:20]

    return {
        "content": [{
            "type": "text",
            "text": json.dumps(top_matches, indent=2)
        }]
    }


def handle_get_endpoint_details(loader: SwaggerLoader, args: Dict[str, Any]) -> Dict[str, Any]:
    """Handle get_endpoint_details tool."""
    path = str(args.get("path", ""))
    method = str(args.get("method", "")).lower()
    service_name = args.get("service_name")

    try:
        doc, name, _ = loader.get_doc(service_name)

        path_obj = doc.paths.get(path)
        if not path_obj:
            return {
                "content": [{
                    "type": "text",
                    "text": f"Path '{path}' not found in '{name}'."
                }],
                "isError": True
            }

        method_obj = path_obj.get(method)
        if not method_obj:
            return {
                "content": [{
                    "type": "text",
                    "text": f"Method '{method}' not found for '{path}' in '{name}'."
                }],
                "isError": True
            }

        # Resolve parameters
        parameters = []
        if method_obj.get("parameters"):
            for param in method_obj["parameters"]:
                resolved_param = resolve_schema(param, doc)
                parameters.append({
                    "name": resolved_param.get("name"),
                    "in": resolved_param.get("in"),
                    "required": resolved_param.get("required", False),
                    "description": resolved_param.get("description"),
                    "schema": resolved_param.get("schema")
                })

        # Resolve request body
        request_body = None
        if method_obj.get("requestBody"):
            request_body = resolve_schema(method_obj["requestBody"], doc)

        # Resolve responses
        responses = {}
        if method_obj.get("responses"):
            for code, response in method_obj["responses"].items():
                responses[code] = resolve_schema(response, doc)

        details = {
            "service": name,
            "summary": method_obj.get("summary"),
            "description": method_obj.get("description"),
            "parameters": parameters,
            "requestBody": request_body,
            "responses": responses
        }

        return {
            "content": [{
                "type": "text",
                "text": json.dumps(details, indent=2)
            }]
        }
    except Exception as e:
        return {
            "content": [{
                "type": "text",
                "text": f"Error getting endpoint details: {str(e)}"
            }],
            "isError": True
        }


def handle_debug_endpoint(loader: SwaggerLoader, args: Dict[str, Any]) -> Dict[str, Any]:
    """Handle debug_endpoint tool."""
    path = str(args.get("path", ""))
    method = str(args.get("method", "")).lower()
    service_name = args.get("service_name")
    path_params = args.get("path_params", {}) or {}
    query_params = args.get("query_params", {}) or {}
    headers = args.get("headers", {}) or {}
    body = args.get("body")

    try:
        doc, _, base_url = loader.get_doc(service_name)

        # 1. Inject Auth Token if available
        cached_token = loader.get_auth_token()
        if cached_token and "Authorization" not in headers:
            headers["Authorization"] = f"Bearer {cached_token}"
            # print(f"[AutoAuth] Injected cached token.", file=sys.stderr)

        # 2. Auto-fill required body parameters
        if method in ["post", "put", "patch"]:
            path_obj = doc.paths.get(path)
            if path_obj and path_obj.get(method) and path_obj[method].get("requestBody"):
                resolved_body = resolve_schema(path_obj[method]["requestBody"], doc)
                content = resolved_body.get("content", {})
                json_schema = content.get("application/json", {}).get("schema")
                if json_schema and json_schema.get("properties"):
                    body = body or {}
                    required = json_schema.get("required", [])
                    for req_field in required:
                        if req_field not in body:
                            prop_schema = json_schema["properties"][req_field]
                            prop_type = prop_schema.get("type")
                            if prop_type == "string":
                                body[req_field] = "test_string"
                            elif prop_type in ("number", "integer"):
                                body[req_field] = 0
                            elif prop_type == "boolean":
                                body[req_field] = False
                            elif prop_type == "array":
                                body[req_field] = []
                            elif prop_type == "object":
                                body[req_field] = {}

        # 3. Build URL with path parameters
        final_path = path
        for key, value in path_params.items():
            final_path = final_path.replace(f"{{{key}}}", str(value))

        url = urljoin(base_url, final_path)

        # 4. Make request
        try:
            response = requests.request(
                method=method.upper(),
                url=url,
                params=query_params,
                headers={"Content-Type": "application/json", **headers},
                json=body,
                timeout=30
            )

            # 5. Capture token from response (heuristic)
            if response.status_code == 200:
                data = response.json() if response.headers.get("content-type", "").startswith("application/json") else response.text
                if isinstance(data, dict):
                    token_candidate = data.get("token") or data.get("accessToken") or data.get("access_token")
                    if not token_candidate and "authorization" in response.headers:
                        token_candidate = response.headers["authorization"]

                    if token_candidate and isinstance(token_candidate, str):
                        raw_token = token_candidate.replace("Bearer ", "", 1).replace("bearer ", "", 1)
                        if len(raw_token) > 20:
                            loader.set_auth_token(raw_token)

            # 6. Return response
            result = {
                "status": response.status_code,
                "statusText": response.reason,
                "headers": dict(response.headers),
                "data": response.json() if response.headers.get("content-type", "").startswith("application/json") else response.text
            }

            return {
                "content": [{
                    "type": "text",
                    "text": json.dumps(result, indent=2)
                }]
            }

        except requests.RequestException as error:
            # Handle 401 Auto-Login Logic
            if hasattr(error, 'response') and error.response is not None and error.response.status_code == 401:
                creds = loader.get_credentials()
                if creds:
                    if "loginPath" in creds:
                        # Option B: Fully Automatic Login (if path is known)
                        try:
                            print(f"[AutoAuth] 401 detected. Attempting auto-login to {creds['loginPath']}...", file=sys.stderr)
                            login_url = urljoin(base_url, creds["loginPath"])
                            login_body = {"username": creds["user"], "password": creds["pass"]}

                            login_res = requests.post(login_url, json=login_body, timeout=30)
                            login_data = login_res.json() if login_res.headers.get("content-type", "").startswith("application/json") else login_res.text

                            token_candidate = None
                            if isinstance(login_data, dict):
                                token_candidate = login_data.get("token") or login_data.get("accessToken") or login_data.get("access_token")

                            if token_candidate:
                                new_token = token_candidate.replace("Bearer ", "", 1).replace("bearer ", "", 1)
                                loader.set_auth_token(new_token)

                                # Retry original request
                                print(f"[AutoAuth] Login success. Retrying original request...", file=sys.stderr)
                                headers["Authorization"] = f"Bearer {new_token}"
                                retry_res = requests.request(
                                    method=method.upper(),
                                    url=url,
                                    params=query_params,
                                    headers={"Content-Type": "application/json", **headers},
                                    json=body,
                                    timeout=30
                                )

                                retry_result = {
                                    "status": retry_res.status_code,
                                    "data": retry_res.json() if retry_res.headers.get("content-type", "").startswith("application/json") else retry_res.text
                                }

                                return {
                                    "content": [{
                                        "type": "text",
                                        "text": json.dumps(retry_result, indent=2)
                                    }]
                                }
                        except Exception as login_err:
                            print(f"[AutoAuth] Auto-login failed: {login_err}", file=sys.stderr)
                    else:
                        # Option A: Hint AI to login
                        return {
                            "content": [{
                                "type": "text",
                                "text": f"Request failed with 401 Unauthorized.\n\n💡 TIP: I have stored credentials (User: {creds['user']}). You can attempt to login by calling the appropriate login endpoint (e.g., POST /auth/login) with these credentials, and I will automatically cache the token for future requests."
                            }],
                            "isError": True
                        }

            # Return error details
            error_msg = f"Request Failed:\nStatus: {error.response.status_code if hasattr(error, 'response') and error.response else 'N/A'}\n"
            if hasattr(error, 'response') and error.response:
                error_msg += f"Data: {json.dumps(error.response.json() if error.response.headers.get('content-type', '').startswith('application/json') else error.response.text, indent=2)}\n"
            error_msg += f"Message: {str(error)}"

            return {
                "content": [{
                    "type": "text",
                    "text": error_msg
                }],
                "isError": True
            }

    except Exception as e:
        return {
            "content": [{
                "type": "text",
                "text": f"Error in debug_endpoint: {str(e)}"
            }],
            "isError": True
        }


def handle_generate_curl(loader: SwaggerLoader, args: Dict[str, Any]) -> Dict[str, Any]:
    """Handle generate_curl tool."""
    path = str(args.get("path", ""))
    method = str(args.get("method", "")).upper()
    service_name = args.get("service_name")
    path_params = args.get("path_params", {}) or {}
    query_params = args.get("query_params", {}) or {}
    headers = args.get("headers", {}) or {}
    body = args.get("body")

    try:
        _, _, base_url = loader.get_doc(service_name)

        # Build URL with path parameters
        final_path = path
        for key, value in path_params.items():
            final_path = final_path.replace(f"{{{key}}}", str(value))

        # Build query string
        url = urljoin(base_url, final_path)
        if query_params:
            query_string = urlencode(query_params)
            url += f"?{query_string}"

        # Build curl command
        parts = [f"curl -X {method} \"{url}\""]

        # Add headers
        all_headers = {"Content-Type": "application/json", **headers}
        for key, value in all_headers.items():
            parts.append(f"-H \"{key}: {value}\"")

        # Add body for applicable methods
        if body and method in ["POST", "PUT", "PATCH"]:
            parts.append(f"-d '{json.dumps(body)}'")

        curl_command = " \\\n  ".join(parts)

        return {
            "content": [{
                "type": "text",
                "text": curl_command
            }]
        }

    except Exception as e:
        return {
            "content": [{
                "type": "text",
                "text": f"Error generating curl command: {str(e)}"
            }],
            "isError": True
        }


# Map tool names to handlers
TOOL_HANDLERS = {
    "list_services": handle_list_services,
    "refresh_docs": handle_refresh_docs,
    "list_endpoints": handle_list_endpoints,
    "search_apis": handle_search_apis,
    "get_endpoint_details": handle_get_endpoint_details,
    "debug_endpoint": handle_debug_endpoint,
    "generate_curl": handle_generate_curl
}