"""
JSON Schema resolution and TypeScript type generation utilities.
"""

from typing import Any, Dict, List, Optional
from ..types.swagger import SwaggerDoc


def capitalize(s: str) -> str:
    """Capitalize the first letter of a string."""
    return s[0].upper() + s[1:] if s else s


def json_schema_to_ts(schema: Optional[Dict[str, Any]], name: str = "Root") -> str:
    """
    Convert JSON Schema to TypeScript type definition.

    Args:
        schema: JSON Schema object
        name: Type name

    Returns:
        TypeScript type definition as string
    """
    if not schema:
        return "any"

    # Handle $ref
    if "$ref" in schema:
        parts = schema["$ref"].split("/")
        return parts[-1]

    schema_type = schema.get("type")

    if schema_type == "string":
        if "enum" in schema:
            return " | ".join(f"'{v}'" for v in schema["enum"])
        return "string"

    elif schema_type in ("number", "integer"):
        return "number"

    elif schema_type == "boolean":
        return "boolean"

    elif schema_type == "array":
        if "items" in schema:
            item_type = json_schema_to_ts(schema["items"], "Item")
            return f"{item_type}[]"
        return "any[]"

    elif schema_type == "object":
        if "properties" not in schema:
            return "Record<string, any>"

        properties = schema["properties"]
        required = schema.get("required", [])

        props = []
        for key, prop in properties.items():
            is_required = key in required
            prop_type = json_schema_to_ts(prop, capitalize(key))
            props.append(f"  {key}{'' if is_required else '?'}: {prop_type};")

        return "{\n" + "\n".join(props) + "\n}"

    else:
        return "any"


def resolve_schema(
    schema: Optional[Dict[str, Any]],
    doc: SwaggerDoc,
    stack: Optional[List[str]] = None
) -> Dict[str, Any]:
    """
    Resolve JSON Schema references within a Swagger document.

    Args:
        schema: JSON Schema object (may contain $ref)
        doc: Swagger document
        stack: Track references to detect cycles

    Returns:
        Resolved schema without $ref
    """
    if not schema:
        return {}

    if stack is None:
        stack = []

    # Handle $ref
    if "$ref" in schema:
        ref = schema["$ref"]
        if ref in stack:
            return {"type": "object", "description": f"[Circular Reference: {ref}]"}

        # Resolve reference path (e.g., "#/components/schemas/User")
        ref_path = ref.replace("#/", "").split("/")
        current: Any = doc.dict()  # Convert to dict for easier traversal

        for part in ref_path:
            if current and isinstance(current, dict):
                current = current.get(part)
            else:
                current = None
                break

        return resolve_schema(current, doc, stack + [ref])

    # Handle arrays
    if schema.get("type") == "array" and "items" in schema:
        return {
            **schema,
            "items": resolve_schema(schema["items"], doc, stack)
        }

    # Handle objects with properties
    if "properties" in schema:
        resolved_props: Dict[str, Any] = {}
        for key, prop in schema["properties"].items():
            resolved_props[key] = resolve_schema(prop, doc, stack)

        return {
            **schema,
            "properties": resolved_props
        }

    # Handle allOf (composition)
    if "allOf" in schema:
        combined: Dict[str, Any] = {}
        for sub_schema in schema["allOf"]:
            resolved = resolve_schema(sub_schema, doc, stack)
            if isinstance(resolved, dict):
                combined.update(resolved)
        return combined

    # Handle anyOf, oneOf (simplified)
    if "anyOf" in schema:
        return {
            **schema,
            "anyOf": [resolve_schema(s, doc, stack) for s in schema["anyOf"]]
        }

    if "oneOf" in schema:
        return {
            **schema,
            "oneOf": [resolve_schema(s, doc, stack) for s in schema["oneOf"]]
        }

    return schema