import os
import inspect # <--- The Magic Library
from .loader import BundleExecutor
from .strategies import NumericalStrategy

class Swarm:
    def __init__(self, *bundles):
        self.agents = []
        
        # --- MAGIC: Auto-detect caller's directory ---
        # 1. Get the previous stack frame (who called Swarm?)
        caller_frame = inspect.stack()[1]
        # 2. Get that file's path (e.g., .../in_vouge/main.py)
        caller_file = caller_frame.filename
        # 3. Get the directory (e.g., .../in_vouge/)
        base_dir = os.path.dirname(os.path.abspath(caller_file))
        # ---------------------------------------------

        for b in bundles:
            target_files = []
            func_name = None

            # Normalize Input (Handle Tuples, Lists, Strings)
            if isinstance(b, tuple): 
                # input: (['path'], 'func_name')
                raw_files = b[0] if isinstance(b[0], list) else [b[0]]
                func_name = b[1]
            elif isinstance(b, list):
                # input: ['path']
                raw_files = b
            elif isinstance(b, str):
                # input: "path"
                raw_files = [b]
            else:
                continue

            # --- PATH RESOLUTION ---
            # Automatically turn "model/pred.py" into "C:/Users/.../model/pred.py"
            clean_files = []
            for f in raw_files:
                if not os.path.isabs(f):
                    clean_files.append(os.path.join(base_dir, f))
                else:
                    clean_files.append(f)
            # -----------------------

            self.agents.append({
                "files": clean_files,
                "func": func_name
            })

    def run(self, input_data, priorities: list = None, sensitivity: float = 1.5):
        # (This part remains exactly the same as before)
        results = []
        for i, agent in enumerate(self.agents):
            try:
                val = BundleExecutor.execute_bundle(
                    agent["files"], 
                    input_data, 
                    function_name=agent["func"]
                )
                results.append(val)
            except Exception as e:
                print(f"⚠️ Agent {i} failed: {e}")
                results.append(None)

        strategy = NumericalStrategy(sensitivity=sensitivity)
        return strategy.aggregate(results, priorities=priorities)