import os
import sys
import importlib.util

class BundleExecutor:
    @staticmethod
    def execute_bundle(file_list, input_data, function_name= None ):
        """
        Takes a list of file paths (bundle), finds the .py file,
        and executes its 'predict' or 'run' function.
        """
        # 1. Find the Python script
        script_path = None
        for path in file_list:
            if path.endswith(".py"):
                script_path = path
                break
        
        if not script_path:
            raise ValueError(f"No .py entry point found in bundle: {file_list}")

        if not os.path.exists(script_path):
            raise FileNotFoundError(f"Script path does not exist: {script_path}")

        # 2. Dynamic Import Magic
        # Create a unique module name so Python doesn't get confused
        module_name = "agent_" + os.path.basename(script_path).replace(".py", "")
        
        try:
            spec = importlib.util.spec_from_file_location(module_name, script_path)
            module = importlib.util.module_from_spec(spec)
            
            # CRITICAL: Add the script's folder to sys.path
            # This allows the script to import its sibling .pkl files naturally
            script_dir = os.path.dirname(os.path.abspath(script_path))
            if script_dir not in sys.path:
                sys.path.append(script_dir)

            # Load the module
            spec.loader.exec_module(module)

            # 3. Execution
            # Check for standard function names
            # 1. If User gave a specific name, try ONLY that
            if function_name:
                if hasattr(module, function_name):
                    func = getattr(module, function_name)
                    return float(func(input_data))
                else:
                    raise ValueError(f"Function '{function_name}' not found in {script_path}")

            # 2. If no name given, try defaults (Backward Compatibility)
            if hasattr(module, "predict"):
                return float(module.predict(input_data))
            elif hasattr(module, "run"):
                return float(module.run(input_data))
            else:
                raise ValueError(f"Script {os.path.basename(script_path)} needs 'predict(data)' or '{function_name}'")
                    
        except Exception as e:
            raise RuntimeError(f"Error executing {os.path.basename(script_path)}: {e}")