import numpy as np

class NumericalStrategy:
    def __init__(self, sensitivity=1.5):
        self.sensitivity = sensitivity # Z-Score threshold (Standard Deviations)

    def aggregate(self, values: list, priorities: list = None):
        # 1. Sanitize Data
        clean_values = []
        clean_weights = []
        
        for i, val in enumerate(values):
            if val is not None and isinstance(val, (int, float)) and not np.isnan(val):
                # Use priority if available, else default to 1.0
                w = priorities[i] if (priorities and i < len(priorities)) else 1.0
                clean_values.append(val)
                clean_weights.append(w)
        
        if not clean_values:
            return {"final_output": None, "analytics": {"error": "No valid numeric outputs"}}

        data_arr = np.array(clean_values)
        weight_arr = np.array(clean_weights)

        # 2. Outlier Removal (Z-Score)
        # Only apply if we have enough data (at least 3 models) to calculate variance
        if len(data_arr) >= 3 and np.std(data_arr) > 0:
            z_scores = np.abs((data_arr - np.mean(data_arr)) / np.std(data_arr))
            mask = z_scores < self.sensitivity
            
            filtered_data = data_arr[mask]
            filtered_weights = weight_arr[mask]
        else:
            filtered_data = data_arr
            filtered_weights = weight_arr

        # 3. Final Calculation
        if len(filtered_data) == 0:
            # Fallback if aggressive filtering removed everything
            final_val = np.average(data_arr, weights=weight_arr)
        else:
            final_val = np.average(filtered_data, weights=filtered_weights)

        return {
            "final_output": float(final_val),
            "analytics": {
                "raw_inputs": clean_values,
                "filtered_inputs": filtered_data.tolist(),
                "std_dev": float(np.std(data_arr)),
                "min": float(np.min(data_arr)),
                "max": float(np.max(data_arr))
            }
        }