// Server session client - used by server_mode.rs for connecting to `sy --server`
// Some methods are reserved for future features (chunked transfer, graceful close)
#![allow(dead_code)]

use anyhow::{Context, Result};
use std::path::Path;
use std::process::Stdio;
use tokio::io::{AsyncReadExt, AsyncWriteExt};
use tokio::process::{Child, Command};

#[cfg(unix)]
use tokio::net::UnixStream;

#[cfg(unix)]
use crate::server::daemon::{read_set_root_ack, write_set_root, MSG_PING, MSG_PONG};
use crate::server::protocol::{
    self, ChecksumReq, ChecksumResp, Decision, DeltaData, DeltaOp, FileData, FileDone, FileList,
    FileListAck, FileListEntry, Hello, MessageType, MkdirBatch, MkdirBatchAck, SymlinkBatch,
    SymlinkBatchAck, SymlinkEntry, HELLO_FLAG_PULL, PROTOCOL_VERSION,
};
use crate::ssh::config::SshConfig;

/// Manages the client-side connection to a remote sy --server instance
pub struct ServerSession {
    child: Child,
    stdin: tokio::process::ChildStdin,
    stdout: tokio::process::ChildStdout,
}

impl ServerSession {
    pub async fn connect_ssh(config: &SshConfig, remote_path: &Path) -> Result<Self> {
        let mut cmd = Command::new("ssh");

        cmd.arg(&config.hostname);
        if !config.user.is_empty() {
            cmd.arg("-l").arg(&config.user);
        }

        if config.port != 22 {
            cmd.arg("-p").arg(config.port.to_string());
        }

        for key in &config.identity_file {
            cmd.arg("-i").arg(key);
        }

        // Remote command: sy --server <remote_path>
        cmd.arg("sy");
        cmd.arg("--server");
        cmd.arg(remote_path);

        cmd.stdin(Stdio::piped());
        cmd.stdout(Stdio::piped());
        cmd.stderr(Stdio::inherit());

        let mut child = cmd.spawn().context("Failed to spawn SSH process")?;

        let stdin = child.stdin.take().context("Failed to open stdin")?;
        let stdout = child.stdout.take().context("Failed to open stdout")?;

        let mut session = Self {
            child,
            stdin,
            stdout,
        };

        session.handshake().await?;

        Ok(session)
    }

    pub async fn connect_local(remote_path: &Path) -> Result<Self> {
        let exe = std::env::current_exe()?;
        let mut cmd = Command::new(exe);
        cmd.arg("--server");
        cmd.arg(remote_path);

        cmd.stdin(Stdio::piped());
        cmd.stdout(Stdio::piped());
        cmd.stderr(Stdio::inherit());

        let mut child = cmd.spawn().context("Failed to spawn sy process")?;

        let stdin = child.stdin.take().context("Failed to open stdin")?;
        let stdout = child.stdout.take().context("Failed to open stdout")?;

        let mut session = Self {
            child,
            stdin,
            stdout,
        };

        session.handshake().await?;

        Ok(session)
    }

    async fn handshake(&mut self) -> Result<()> {
        let hello = Hello {
            version: PROTOCOL_VERSION,
            flags: 0,
            capabilities: vec![],
        };

        hello.write(&mut self.stdin).await?;
        self.stdin.flush().await?;

        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server handshake error: {}", err.message));
        }

        if type_byte != MessageType::Hello as u8 {
            return Err(anyhow::anyhow!("Expected HELLO, got 0x{:02X}", type_byte));
        }

        let resp = Hello::read(&mut self.stdout).await?;

        if resp.version != PROTOCOL_VERSION {
            return Err(anyhow::anyhow!("Version mismatch: server {}", resp.version));
        }

        Ok(())
    }

    // =========================================================================
    // FILE_LIST
    // =========================================================================

    pub async fn send_file_list(&mut self, entries: Vec<FileListEntry>) -> Result<()> {
        let list = FileList { entries };
        list.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    pub async fn read_ack(&mut self) -> Result<FileListAck> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::FileListAck as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_LIST_ACK, got 0x{:02X}",
                type_byte
            ));
        }

        FileListAck::read(&mut self.stdout).await
    }

    // =========================================================================
    // MKDIR_BATCH
    // =========================================================================

    pub async fn send_mkdir_batch(&mut self, paths: Vec<String>) -> Result<()> {
        let batch = MkdirBatch { paths };
        batch.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    pub async fn read_mkdir_ack(&mut self) -> Result<MkdirBatchAck> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::MkdirBatchAck as u8 {
            return Err(anyhow::anyhow!(
                "Expected MKDIR_BATCH_ACK, got 0x{:02X}",
                type_byte
            ));
        }

        MkdirBatchAck::read(&mut self.stdout).await
    }

    // =========================================================================
    // SYMLINK_BATCH
    // =========================================================================

    pub async fn send_symlink_batch(&mut self, entries: Vec<SymlinkEntry>) -> Result<()> {
        let batch = SymlinkBatch { entries };
        batch.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    pub async fn read_symlink_ack(&mut self) -> Result<SymlinkBatchAck> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::SymlinkBatchAck as u8 {
            return Err(anyhow::anyhow!(
                "Expected SYMLINK_BATCH_ACK, got 0x{:02X}",
                type_byte
            ));
        }

        SymlinkBatchAck::read(&mut self.stdout).await
    }

    // =========================================================================
    // FILE_DATA
    // =========================================================================

    pub async fn send_file_data(&mut self, index: u32, offset: u64, data: Vec<u8>) -> Result<()> {
        let file_data = FileData {
            index,
            offset,
            flags: 0,
            data,
        };
        file_data.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    /// Send file data with flags (e.g., compressed), without flushing - use flush() after batch
    pub async fn send_file_data_no_flush(
        &mut self,
        index: u32,
        offset: u64,
        data: Vec<u8>,
    ) -> Result<()> {
        let file_data = FileData {
            index,
            offset,
            flags: 0,
            data,
        };
        file_data.write(&mut self.stdin).await?;
        Ok(())
    }

    /// Send file data with explicit flags, without flushing
    pub async fn send_file_data_with_flags(
        &mut self,
        index: u32,
        offset: u64,
        flags: u8,
        data: Vec<u8>,
    ) -> Result<()> {
        let file_data = FileData {
            index,
            offset,
            flags,
            data,
        };
        file_data.write(&mut self.stdin).await?;
        Ok(())
    }

    /// Flush the write buffer
    pub async fn flush(&mut self) -> Result<()> {
        self.stdin.flush().await?;
        Ok(())
    }

    pub async fn read_file_done(&mut self) -> Result<FileDone> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::FileDone as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_DONE, got 0x{:02X}",
                type_byte
            ));
        }

        FileDone::read(&mut self.stdout).await
    }

    // =========================================================================
    // DELTA SYNC
    // =========================================================================

    /// Request block checksums for a file (for delta sync)
    pub async fn send_checksum_req(&mut self, index: u32, block_size: u32) -> Result<()> {
        let req = ChecksumReq { index, block_size };
        req.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    /// Request block checksums without flushing - use flush() after batch
    pub async fn send_checksum_req_no_flush(&mut self, index: u32, block_size: u32) -> Result<()> {
        let req = ChecksumReq { index, block_size };
        req.write(&mut self.stdin).await?;
        Ok(())
    }

    /// Read checksum response
    pub async fn read_checksum_resp(&mut self) -> Result<ChecksumResp> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::ChecksumResp as u8 {
            return Err(anyhow::anyhow!(
                "Expected CHECKSUM_RESP, got 0x{:02X}",
                type_byte
            ));
        }

        ChecksumResp::read(&mut self.stdout).await
    }

    /// Send delta data (for updating existing file)
    pub async fn send_delta_data(
        &mut self,
        index: u32,
        flags: u8,
        ops: Vec<DeltaOp>,
    ) -> Result<()> {
        let delta = DeltaData { index, flags, ops };
        delta.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    /// Send delta data without flushing
    pub async fn send_delta_data_no_flush(
        &mut self,
        index: u32,
        flags: u8,
        ops: Vec<DeltaOp>,
    ) -> Result<()> {
        let delta = DeltaData { index, flags, ops };
        delta.write(&mut self.stdin).await?;
        Ok(())
    }

    // =========================================================================
    // PULL MODE (client receives files from server)
    // =========================================================================

    /// Connect to SSH server in PULL mode (server sends files to client)
    pub async fn connect_ssh_pull(config: &SshConfig, remote_path: &Path) -> Result<Self> {
        let mut cmd = Command::new("ssh");

        cmd.arg(&config.hostname);
        if !config.user.is_empty() {
            cmd.arg("-l").arg(&config.user);
        }

        if config.port != 22 {
            cmd.arg("-p").arg(config.port.to_string());
        }

        for key in &config.identity_file {
            cmd.arg("-i").arg(key);
        }

        cmd.arg("sy");
        cmd.arg("--server");
        cmd.arg(remote_path);

        cmd.stdin(Stdio::piped());
        cmd.stdout(Stdio::piped());
        cmd.stderr(Stdio::inherit());

        let mut child = cmd.spawn().context("Failed to spawn SSH process")?;

        let stdin = child.stdin.take().context("Failed to open stdin")?;
        let stdout = child.stdout.take().context("Failed to open stdout")?;

        let mut session = Self {
            child,
            stdin,
            stdout,
        };

        session.handshake_pull().await?;

        Ok(session)
    }

    /// Connect to local sy --server in PULL mode
    pub async fn connect_local_pull(remote_path: &Path) -> Result<Self> {
        let exe = std::env::current_exe()?;
        let mut cmd = Command::new(exe);
        cmd.arg("--server");
        cmd.arg(remote_path);

        cmd.stdin(Stdio::piped());
        cmd.stdout(Stdio::piped());
        cmd.stderr(Stdio::inherit());

        let mut child = cmd.spawn().context("Failed to spawn sy process")?;

        let stdin = child.stdin.take().context("Failed to open stdin")?;
        let stdout = child.stdout.take().context("Failed to open stdout")?;

        let mut session = Self {
            child,
            stdin,
            stdout,
        };

        session.handshake_pull().await?;

        Ok(session)
    }

    /// Handshake with PULL flag set
    async fn handshake_pull(&mut self) -> Result<()> {
        let hello = Hello {
            version: PROTOCOL_VERSION,
            flags: HELLO_FLAG_PULL,
            capabilities: vec![],
        };

        hello.write(&mut self.stdin).await?;
        self.stdin.flush().await?;

        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server handshake error: {}", err.message));
        }

        if type_byte != MessageType::Hello as u8 {
            return Err(anyhow::anyhow!("Expected HELLO, got 0x{:02X}", type_byte));
        }

        let resp = Hello::read(&mut self.stdout).await?;

        if resp.version != PROTOCOL_VERSION {
            return Err(anyhow::anyhow!("Version mismatch: server {}", resp.version));
        }

        Ok(())
    }

    /// Read MKDIR_BATCH from server (PULL mode) - server always sends this
    pub async fn read_mkdir_batch(&mut self) -> Result<MkdirBatch> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::MkdirBatch as u8 {
            return Err(anyhow::anyhow!(
                "Expected MKDIR_BATCH, got 0x{:02X}",
                type_byte
            ));
        }

        let batch = MkdirBatch::read(&mut self.stdout).await?;
        Ok(batch)
    }

    /// Send MKDIR_BATCH_ACK to server (PULL mode)
    pub async fn send_mkdir_batch_ack(
        &mut self,
        created: u32,
        failed: Vec<(String, String)>,
    ) -> Result<()> {
        let ack = MkdirBatchAck { created, failed };
        ack.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    /// Read FILE_LIST from server (PULL mode)
    pub async fn read_file_list(&mut self) -> Result<FileList> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::FileList as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_LIST, got 0x{:02X}",
                type_byte
            ));
        }

        FileList::read(&mut self.stdout).await
    }

    /// Send FILE_LIST_ACK to server (PULL mode)
    pub async fn send_file_list_ack(&mut self, decisions: Vec<Decision>) -> Result<()> {
        let ack = FileListAck { decisions };
        ack.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    /// Read FILE_DATA from server (PULL mode)
    pub async fn read_file_data(&mut self) -> Result<Option<FileData>> {
        let _len = self.stdout.read_u32().await?;
        let type_byte = self.stdout.read_u8().await?;

        // Server might send SYMLINK_BATCH if done with files
        if type_byte == MessageType::SymlinkBatch as u8 {
            return Ok(None);
        }

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.stdout).await?;
            return Err(anyhow::anyhow!("Server error: {}", err.message));
        }

        if type_byte != MessageType::FileData as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_DATA or SYMLINK_BATCH, got 0x{:02X}",
                type_byte
            ));
        }

        let data = FileData::read(&mut self.stdout).await?;
        Ok(Some(data))
    }

    /// Send FILE_DONE to server (PULL mode)
    pub async fn send_file_done(&mut self, index: u32, status: u8) -> Result<()> {
        let done = FileDone {
            index,
            status,
            checksum: vec![],
        };
        done.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    /// Read SYMLINK_BATCH from server (PULL mode) - assumes type byte already read
    pub async fn read_symlink_batch_body(&mut self) -> Result<SymlinkBatch> {
        SymlinkBatch::read(&mut self.stdout).await
    }

    /// Send SYMLINK_BATCH_ACK to server (PULL mode)
    pub async fn send_symlink_batch_ack(
        &mut self,
        created: u32,
        failed: Vec<(String, String)>,
    ) -> Result<()> {
        let ack = SymlinkBatchAck { created, failed };
        ack.write(&mut self.stdin).await?;
        self.stdin.flush().await?;
        Ok(())
    }

    // =========================================================================
    // Lifecycle
    // =========================================================================

    /// Close the session gracefully
    pub async fn close(mut self) -> Result<()> {
        drop(self.stdin);
        let _ = self.child.wait().await;
        Ok(())
    }
}

// =============================================================================
// DaemonSession - Client for connecting to daemon via Unix socket
// =============================================================================

/// Session connected to a daemon via Unix socket
///
/// This provides the same interface as ServerSession but connects via Unix socket
/// instead of spawning an SSH process. Use with SSH socket forwarding for fast
/// repeated syncs:
///
/// ```bash
/// # Start daemon on remote
/// sy --daemon --socket ~/.sy/daemon.sock
///
/// # Forward socket via SSH
/// ssh -L /tmp/sy.sock:~/.sy/daemon.sock user@host -N &
///
/// # Connect to daemon
/// let session = DaemonSession::connect("/tmp/sy.sock", "/remote/path").await?;
/// ```
#[cfg(unix)]
pub struct DaemonSession {
    reader: tokio::net::unix::OwnedReadHalf,
    writer: tokio::net::unix::OwnedWriteHalf,
}

#[cfg(unix)]
impl DaemonSession {
    /// Connect to a daemon via Unix socket and set the root path
    pub async fn connect(socket_path: &str, remote_path: &Path) -> Result<Self> {
        let stream = UnixStream::connect(socket_path)
            .await
            .with_context(|| format!("Failed to connect to daemon at {}", socket_path))?;

        let (reader, writer) = stream.into_split();
        let mut session = Self { reader, writer };

        // Handshake
        session.handshake().await?;

        // Set root path
        let path_str = remote_path.to_string_lossy();
        write_set_root(&mut session.writer, &path_str).await?;

        if !read_set_root_ack(&mut session.reader).await? {
            return Err(anyhow::anyhow!(
                "Daemon rejected root path: {}",
                remote_path.display()
            ));
        }

        Ok(session)
    }

    /// Connect to a daemon in PULL mode (daemon sends files to client)
    pub async fn connect_pull(socket_path: &str, remote_path: &Path) -> Result<Self> {
        let stream = UnixStream::connect(socket_path)
            .await
            .with_context(|| format!("Failed to connect to daemon at {}", socket_path))?;

        let (reader, writer) = stream.into_split();
        let mut session = Self { reader, writer };

        // Handshake with PULL flag
        session.handshake_pull().await?;

        // Set root path
        let path_str = remote_path.to_string_lossy();
        write_set_root(&mut session.writer, &path_str).await?;

        if !read_set_root_ack(&mut session.reader).await? {
            return Err(anyhow::anyhow!(
                "Daemon rejected root path: {}",
                remote_path.display()
            ));
        }

        Ok(session)
    }

    async fn handshake(&mut self) -> Result<()> {
        let hello = Hello {
            version: PROTOCOL_VERSION,
            flags: 0,
            capabilities: vec![],
        };

        hello.write(&mut self.writer).await?;
        self.writer.flush().await?;

        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon handshake error: {}", err.message));
        }

        if type_byte != MessageType::Hello as u8 {
            return Err(anyhow::anyhow!("Expected HELLO, got 0x{:02X}", type_byte));
        }

        let resp = Hello::read(&mut self.reader).await?;

        if resp.version != PROTOCOL_VERSION {
            return Err(anyhow::anyhow!("Version mismatch: daemon {}", resp.version));
        }

        Ok(())
    }

    async fn handshake_pull(&mut self) -> Result<()> {
        let hello = Hello {
            version: PROTOCOL_VERSION,
            flags: HELLO_FLAG_PULL,
            capabilities: vec![],
        };

        hello.write(&mut self.writer).await?;
        self.writer.flush().await?;

        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon handshake error: {}", err.message));
        }

        if type_byte != MessageType::Hello as u8 {
            return Err(anyhow::anyhow!("Expected HELLO, got 0x{:02X}", type_byte));
        }

        let resp = Hello::read(&mut self.reader).await?;

        if resp.version != PROTOCOL_VERSION {
            return Err(anyhow::anyhow!("Version mismatch: daemon {}", resp.version));
        }

        Ok(())
    }

    // =========================================================================
    // Keepalive
    // =========================================================================

    /// Send a PING to check if daemon is alive
    pub async fn ping(&mut self) -> Result<()> {
        self.writer.write_u32(0).await?;
        self.writer.write_u8(MSG_PING).await?;
        self.writer.flush().await?;

        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte != MSG_PONG {
            return Err(anyhow::anyhow!("Expected PONG, got 0x{:02X}", type_byte));
        }

        Ok(())
    }

    // =========================================================================
    // FILE_LIST
    // =========================================================================

    pub async fn send_file_list(&mut self, entries: Vec<FileListEntry>) -> Result<()> {
        let list = FileList { entries };
        list.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn read_ack(&mut self) -> Result<FileListAck> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::FileListAck as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_LIST_ACK, got 0x{:02X}",
                type_byte
            ));
        }

        FileListAck::read(&mut self.reader).await
    }

    // =========================================================================
    // MKDIR_BATCH
    // =========================================================================

    pub async fn send_mkdir_batch(&mut self, paths: Vec<String>) -> Result<()> {
        let batch = MkdirBatch { paths };
        batch.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn read_mkdir_ack(&mut self) -> Result<MkdirBatchAck> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::MkdirBatchAck as u8 {
            return Err(anyhow::anyhow!(
                "Expected MKDIR_BATCH_ACK, got 0x{:02X}",
                type_byte
            ));
        }

        MkdirBatchAck::read(&mut self.reader).await
    }

    // =========================================================================
    // SYMLINK_BATCH
    // =========================================================================

    pub async fn send_symlink_batch(&mut self, entries: Vec<SymlinkEntry>) -> Result<()> {
        let batch = SymlinkBatch { entries };
        batch.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn read_symlink_ack(&mut self) -> Result<SymlinkBatchAck> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::SymlinkBatchAck as u8 {
            return Err(anyhow::anyhow!(
                "Expected SYMLINK_BATCH_ACK, got 0x{:02X}",
                type_byte
            ));
        }

        SymlinkBatchAck::read(&mut self.reader).await
    }

    // =========================================================================
    // FILE_DATA
    // =========================================================================

    pub async fn send_file_data(&mut self, index: u32, offset: u64, data: Vec<u8>) -> Result<()> {
        let file_data = FileData {
            index,
            offset,
            flags: 0,
            data,
        };
        file_data.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn send_file_data_no_flush(
        &mut self,
        index: u32,
        offset: u64,
        data: Vec<u8>,
    ) -> Result<()> {
        let file_data = FileData {
            index,
            offset,
            flags: 0,
            data,
        };
        file_data.write(&mut self.writer).await?;
        Ok(())
    }

    pub async fn send_file_data_with_flags(
        &mut self,
        index: u32,
        offset: u64,
        flags: u8,
        data: Vec<u8>,
    ) -> Result<()> {
        let file_data = FileData {
            index,
            offset,
            flags,
            data,
        };
        file_data.write(&mut self.writer).await?;
        Ok(())
    }

    pub async fn flush(&mut self) -> Result<()> {
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn read_file_done(&mut self) -> Result<FileDone> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::FileDone as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_DONE, got 0x{:02X}",
                type_byte
            ));
        }

        FileDone::read(&mut self.reader).await
    }

    // =========================================================================
    // DELTA SYNC
    // =========================================================================

    pub async fn send_checksum_req(&mut self, index: u32, block_size: u32) -> Result<()> {
        let req = ChecksumReq { index, block_size };
        req.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn send_checksum_req_no_flush(&mut self, index: u32, block_size: u32) -> Result<()> {
        let req = ChecksumReq { index, block_size };
        req.write(&mut self.writer).await?;
        Ok(())
    }

    pub async fn read_checksum_resp(&mut self) -> Result<ChecksumResp> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::ChecksumResp as u8 {
            return Err(anyhow::anyhow!(
                "Expected CHECKSUM_RESP, got 0x{:02X}",
                type_byte
            ));
        }

        ChecksumResp::read(&mut self.reader).await
    }

    pub async fn send_delta_data(
        &mut self,
        index: u32,
        flags: u8,
        ops: Vec<DeltaOp>,
    ) -> Result<()> {
        let delta = DeltaData { index, flags, ops };
        delta.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn send_delta_data_no_flush(
        &mut self,
        index: u32,
        flags: u8,
        ops: Vec<DeltaOp>,
    ) -> Result<()> {
        let delta = DeltaData { index, flags, ops };
        delta.write(&mut self.writer).await?;
        Ok(())
    }

    // =========================================================================
    // PULL MODE
    // =========================================================================

    pub async fn read_mkdir_batch(&mut self) -> Result<MkdirBatch> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::MkdirBatch as u8 {
            return Err(anyhow::anyhow!(
                "Expected MKDIR_BATCH, got 0x{:02X}",
                type_byte
            ));
        }

        MkdirBatch::read(&mut self.reader).await
    }

    pub async fn send_mkdir_batch_ack(
        &mut self,
        created: u32,
        failed: Vec<(String, String)>,
    ) -> Result<()> {
        let ack = MkdirBatchAck { created, failed };
        ack.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn read_file_list(&mut self) -> Result<FileList> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::FileList as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_LIST, got 0x{:02X}",
                type_byte
            ));
        }

        FileList::read(&mut self.reader).await
    }

    pub async fn send_file_list_ack(&mut self, decisions: Vec<Decision>) -> Result<()> {
        let ack = FileListAck { decisions };
        ack.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn read_file_data(&mut self) -> Result<Option<FileData>> {
        let _len = self.reader.read_u32().await?;
        let type_byte = self.reader.read_u8().await?;

        if type_byte == MessageType::SymlinkBatch as u8 {
            return Ok(None);
        }

        if type_byte == MessageType::Error as u8 {
            let err = protocol::ErrorMessage::read(&mut self.reader).await?;
            return Err(anyhow::anyhow!("Daemon error: {}", err.message));
        }

        if type_byte != MessageType::FileData as u8 {
            return Err(anyhow::anyhow!(
                "Expected FILE_DATA or SYMLINK_BATCH, got 0x{:02X}",
                type_byte
            ));
        }

        let data = FileData::read(&mut self.reader).await?;
        Ok(Some(data))
    }

    pub async fn send_file_done(&mut self, index: u32, status: u8) -> Result<()> {
        let done = FileDone {
            index,
            status,
            checksum: vec![],
        };
        done.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    pub async fn read_symlink_batch_body(&mut self) -> Result<SymlinkBatch> {
        SymlinkBatch::read(&mut self.reader).await
    }

    pub async fn send_symlink_batch_ack(
        &mut self,
        created: u32,
        failed: Vec<(String, String)>,
    ) -> Result<()> {
        let ack = SymlinkBatchAck { created, failed };
        ack.write(&mut self.writer).await?;
        self.writer.flush().await?;
        Ok(())
    }

    // =========================================================================
    // Lifecycle
    // =========================================================================

    /// Close the session gracefully
    pub async fn close(self) -> Result<()> {
        // Dropping the reader/writer closes the connection
        drop(self.reader);
        drop(self.writer);
        Ok(())
    }
}
