from typing import Dict
import asyncio

from sycommon.logging.kafka_log import SYLogger
from sycommon.rabbitmq.rabbitmq_client import RabbitMQClient
from sycommon.rabbitmq.rabbitmq_service_core import RabbitMQCoreService

logger = SYLogger


class RabbitMQClientManager(RabbitMQCoreService):
    """
    RabbitMQ客户端管理类 - 负责客户端的创建、获取、重连、资源清理
    """
    # 客户端存储
    _clients: Dict[str, RabbitMQClient] = {}
    _init_locks: Dict[str, asyncio.Lock] = {}

    # 模式标记
    _has_listeners: bool = False
    _has_senders: bool = False

    @classmethod
    def set_mode_flags(cls, has_listeners: bool = False, has_senders: bool = False) -> None:
        """设置模式标记（是否有监听器/发送器）"""
        cls._has_listeners = has_listeners
        cls._has_senders = has_senders

    @classmethod
    async def _clean_client_resources(cls, client: RabbitMQClient) -> None:
        """清理客户端无效资源"""
        try:
            # 先停止消费
            if client._consumer_tag:
                await client.stop_consuming()
            logger.debug("客户端无效资源清理完成（单通道无需归还）")
        except Exception as e:
            logger.warning(f"释放客户端无效资源失败: {str(e)}")
        finally:
            # 强制重置客户端状态
            client._channel = None
            client._channel_conn = None
            client._exchange = None
            client._queue = None
            client._consumer_tag = None

    @classmethod
    async def _reconnect_client(cls, client_name: str, client: RabbitMQClient) -> bool:
        """客户端重连"""
        if cls._is_shutdown or not (cls._connection_pool and await cls._connection_pool.is_alive):
            return False

        # 重连冷却
        await asyncio.sleep(cls.RECONNECT_INTERVAL)

        try:
            # 清理旧资源
            await cls._clean_client_resources(client)

            # 执行重连
            await client.connect()

            # 验证重连结果
            if await client.is_connected:
                logger.info(f"客户端 '{client_name}' 重连成功")
                return True
            else:
                logger.warning(f"客户端 '{client_name}' 重连失败：资源未完全初始化")
                return False
        except Exception as e:
            logger.error(f"客户端 '{client_name}' 重连失败: {str(e)}", exc_info=True)
            return False

    @classmethod
    async def _create_client(cls, queue_name: str, **kwargs) -> RabbitMQClient:
        """创建客户端实例"""
        if cls._is_shutdown:
            raise RuntimeError("RabbitMQService已关闭，无法创建客户端")

        # 等待连接池就绪
        await cls.wait_for_pool_ready()

        app_name = kwargs.get('app_name', cls._config.get(
            "APP_NAME", "")) if cls._config else ""
        # 纯发送器场景，强制不创建队列
        is_sender_only = not cls._has_listeners and cls._has_senders

        # 发送器场景下强制禁用队列创建
        create_if_not_exists = False if is_sender_only else cls._has_listeners

        processed_queue_name = queue_name
        # 只有非纯发送器场景才处理队列名称拼接
        if not is_sender_only and create_if_not_exists and processed_queue_name and app_name:
            if not processed_queue_name.endswith(f".{app_name}"):
                processed_queue_name = f"{processed_queue_name}.{app_name}"
                logger.info(f"监听器队列名称自动拼接app-name: {processed_queue_name}")
            else:
                logger.info(f"监听器队列已包含app-name: {processed_queue_name}")

        logger.info(
            f"创建客户端 - 队列: {processed_queue_name if not is_sender_only else 'N/A'}, "
            f"纯发送器模式: {is_sender_only}, "
            f"允许创建队列: {create_if_not_exists}"
        )

        # 创建客户端实例
        client = RabbitMQClient(
            connection_pool=cls._connection_pool,
            exchange_name=cls._config.get(
                'exchange_name', "system.topic.exchange"),
            exchange_type=kwargs.get('exchange_type', "topic"),
            # 纯发送器场景下队列名称设为None
            queue_name=None if is_sender_only else processed_queue_name,
            routing_key=kwargs.get(
                'routing_key',
                f"{queue_name.split('.')[0]}.#" if queue_name and not is_sender_only else "#"
            ),
            durable=kwargs.get('durable', True),
            auto_delete=kwargs.get('auto_delete', False),
            auto_parse_json=kwargs.get('auto_parse_json', True),
            create_if_not_exists=create_if_not_exists,
            prefetch_count=kwargs.get('prefetch_count', 2),
        )

        # 连接客户端
        await client.connect()

        return client

    @classmethod
    async def get_client(
        cls,
        client_name: str = "default", **kwargs
    ) -> RabbitMQClient:
        """获取或创建RabbitMQ客户端"""
        if cls._is_shutdown:
            raise RuntimeError("RabbitMQService已关闭，无法获取客户端")

        # 等待连接池就绪
        await cls.wait_for_pool_ready()

        # 确保锁存在
        if client_name not in cls._init_locks:
            cls._init_locks[client_name] = asyncio.Lock()

        async with cls._init_locks[client_name]:
            # 如果客户端已存在且连接有效，直接返回
            if client_name in cls._clients:
                client = cls._clients[client_name]
                is_sender_only = not cls._has_listeners and cls._has_senders

                if await client.is_connected:
                    # 只有非纯发送器场景才检查队列初始化状态
                    if not is_sender_only and not client._queue and cls._has_listeners:
                        logger.info(f"客户端 '{client_name}' 队列未初始化，重新连接")
                        client.create_if_not_exists = True
                        await client.connect()
                    return client
                else:
                    logger.info(f"客户端 '{client_name}' 连接已断开，重新连接")
                    if not is_sender_only:
                        client.create_if_not_exists = cls._has_listeners
                    await client.connect()
                    return client

            # 创建新客户端
            initial_queue_name = kwargs.pop('queue_name', '')
            is_sender_only = not cls._has_listeners and cls._has_senders

            # 发送器特殊处理
            if is_sender_only:
                kwargs['create_if_not_exists'] = False
                client = await cls._create_client(
                    "",  # 空队列名
                    app_name=cls._config.get("APP_NAME", ""),
                    **kwargs
                )
                cls._clients[client_name] = client
                return client

            # 监听器逻辑
            kwargs['create_if_not_exists'] = True
            client = await cls._create_client(
                initial_queue_name,
                app_name=cls._config.get("APP_NAME", ""),
                **kwargs
            )

            # 验证队列创建
            if not is_sender_only and not client._queue:
                logger.error(f"队列 '{initial_queue_name}' 创建失败，尝试重新创建")
                client.create_if_not_exists = True
                await client.connect()
                if not client._queue:
                    raise Exception(f"无法创建队列 '{initial_queue_name}'")

            cls._clients[client_name] = client
            return client

    @classmethod
    async def shutdown_clients(cls, timeout: float = 15.0) -> None:
        """关闭所有客户端"""
        # 关闭所有客户端
        for client in cls._clients.values():
            try:
                await client.close()
            except Exception as e:
                logger.error(f"关闭客户端失败: {str(e)}")

        # 清理客户端状态
        cls._clients.clear()
        cls._init_locks.clear()
