import sys
import traceback
import aiohttp
import asyncio
from typing import Optional
from datetime import datetime
from urllib.parse import urlparse, parse_qs, urlencode, urlunparse
from sycommon.config.Config import Config
from sycommon.logging.kafka_log import SYLogger


async def send_wechat_markdown_msg(
    content: str,
    webhook: str = None
) -> Optional[dict]:
    """
    异步发送企业微信Markdown格式的WebHook消息
    """
    # 构造请求体（Markdown格式）
    payload = {
        "msgtype": "markdown",
        "markdown": {
            "content": content
        }
    }

    try:
        # 使用 json 参数自动处理序列化和 Content-Type
        async with aiohttp.ClientSession() as session:
            async with session.post(
                url=webhook,
                json=payload,
                timeout=aiohttp.ClientTimeout(total=10)
            ) as response:
                response_data = await response.json()

                if response.status == 200 and response_data.get("errcode") == 0:
                    SYLogger.info(f"消息发送成功: {response_data}")
                    return response_data
                else:
                    SYLogger.warning(
                        f"消息发送失败 - 状态码: {response.status}, 响应: {response_data}")
                    return None
    except Exception as e:
        SYLogger.error(f"发送企业微信消息异常: {str(e)}")
        return None


async def send_webhook(error_info: dict = None, webhook: str = None):
    """
    发送服务启动结果的企业微信通知
    """
    # 获取服务名和环境（增加默认值保护）
    try:
        config = Config().config
        service_name = config.get('Name', "未知服务")
        env = config.get('Nacos', {}).get('namespaceId', '未知环境')
        # 注意：这里使用了大写开头的 WebHook，请确保配置文件中键名一致
        webHook = config.get('llm', {}).get('WebHook')
    except Exception as e:
        service_name = "未知服务"
        env = "未知环境"
        webHook = None
        SYLogger.warning(f"读取配置失败: {str(e)}")

    start_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S')

    # 如果没有错误信息，就不发送失败告警
    if not error_info:
        return

    # 启动失败的通知内容（包含详细错误信息）
    error_type = error_info.get("error_type", "未知错误")
    error_msg = error_info.get("error_msg", "无错误信息")
    stack_trace = error_info.get("stack_trace", "无堆栈信息")[:1000]  # 适当增加长度限制
    elapsed_time = error_info.get("elapsed_time", 0)

    markdown_content = f"""### {service_name}服务启动失败告警 ⚠️
> 环境: <font color="warning">{env}</font>
> 启动时间: <font color="comment">{start_time}</font>
> 耗时: <font color="comment">{elapsed_time:.2f}秒</font>
> 错误类型: <font color="danger">{error_type}</font>
> 错误信息: <font color="danger">{error_msg}</font>
> 错误堆栈: {stack_trace}"""

    if webhook or webHook:
        result = await send_wechat_markdown_msg(
            content=markdown_content,
            webhook=webhook or webHook
        )
        SYLogger.info(f"通知发送结果: {result}")
    else:
        SYLogger.info("未设置企业微信WebHook，跳过告警发送")


def run(*args, webhook: str = None, **kwargs):
    """
    带企业微信告警的Uvicorn启动监控
    """
    # 判断环境
    try:
        env = Config().config.get('Nacos', {}).get('namespaceId', 'dev')
    except:
        env = 'dev'

    # 如果是生产环境，通常不需要这种启动脚本进行告警干扰，或者你需要根据实际情况调整
    if env == "prod":
        import uvicorn
        uvicorn.run(*args, **kwargs)
        return

    # 记录启动开始时间
    start_time = datetime.now()

    if webhook:
        # 脱敏展示webhook
        try:
            parsed = urlparse(webhook)
            query = parse_qs(parsed.query)
            if 'key' in query and query['key'][0]:
                key = query['key'][0]
                masked_key = key[:8] + "****" if len(key) > 8 else key + "****"
                query['key'] = [masked_key]
                masked_query = urlencode(query, doseq=True)
                masked_webhook = urlunparse(
                    (parsed.scheme, parsed.netloc, parsed.path, parsed.params, masked_query, parsed.fragment))
                SYLogger.info(f"自定义企业微信WebHook: {masked_webhook}")
        except Exception:
            SYLogger.info("自定义Webhook格式解析失败，但会尝试使用")

    # 初始化错误信息
    error_info = None

    try:
        import uvicorn
        # 执行启动
        uvicorn.run(*args, **kwargs)

    except KeyboardInterrupt:
        # 处理用户手动中断
        elapsed = (datetime.now() - start_time).total_seconds()
        SYLogger.info(f"\n{'='*50}")
        SYLogger.info(f"ℹ️  应用被用户手动中断")
        SYLogger.info(f"启动耗时: {elapsed:.2f} 秒")
        SYLogger.info(f"{'='*50}\n")
        # 手动中断不需要错误告警，也不需要 sys.exit(1)
        sys.exit(0)

    except Exception as e:
        # 捕获启动失败异常
        elapsed = (datetime.now() - start_time).total_seconds()
        stack_trace = traceback.format_exc()

        error_info = {
            "error_type": type(e).__name__,
            "error_msg": str(e),
            "stack_trace": stack_trace,
            "elapsed_time": elapsed
        }

        # 打印详细错误
        SYLogger.error(f"\n{'='*50}")
        SYLogger.error(f"🚨 应用启动失败！")
        SYLogger.error(f"失败时间: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
        SYLogger.error(f"错误类型: {type(e).__name__}")
        SYLogger.error(f"错误信息: {str(e)}")
        SYLogger.error(f"\n📝 错误堆栈:")
        SYLogger.error(f"-"*50)
        traceback.print_exc(file=sys.stdout)
        SYLogger.error(f"\n⏱️  启动耗时: {elapsed:.2f} 秒")
        SYLogger.error(f"{'='*50}\n")

    finally:
        # 创建新的事件循环来发送通知，防止与 Uvicorn 的循环冲突
        if error_info:
            try:
                loop = asyncio.new_event_loop()
                asyncio.set_event_loop(loop)
                loop.run_until_complete(send_webhook(
                    error_info=error_info,
                    webhook=webhook
                ))
                loop.close()
            except Exception as e:
                SYLogger.error(f"错误：异步通知发送失败 - {str(e)}")

            # 只有确实有错误时才以状态码 1 退出
            sys.exit(1)
