import os
import smtplib
import asyncio
from typing import List, Optional
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from email.header import Header
from email.mime.base import MIMEBase
from email import encoders
from src.sycommon.config.Config import Config
from src.sycommon.logging.kafka_log import SYLogger


class SYEmail:
    """盛业邮件发送工具类"""

    @staticmethod
    def _get_config() -> dict:
        """获取邮件配置"""
        return Config().config.get("AIEmailConfig", {})

    @staticmethod
    async def send(
        to: List[str],
        subject: str,
        content: str,
        attachments: Optional[List[str]] = None,
        verbose: bool = False
    ) -> bool:
        """
        异步发送邮件

        :param to: 收件人列表
        :param subject: 邮件主题
        :param content: 邮件正文
        :param attachments: 附件路径列表 (可选)
        :param verbose: 是否打印详细日志
        :return: 是否发送成功
        """
        try:
            # 1. 构建邮件对象
            message = SYEmail._build_message(
                to, subject, content, attachments
            )

            # 2. 发送邮件 (网络阻塞型，扔到线程池运行)
            loop = asyncio.get_running_loop()
            success = await loop.run_in_executor(
                None,
                SYEmail._send_smtp,
                message,
                to,
                verbose
            )
            return success

        except Exception as e:
            SYLogger.error(f"❌ 异步邮件调度异常: {e}")
            return False

    @staticmethod
    def _build_message(to: List[str], subject: str, content: str, attachments: Optional[List[str]] = None):
        """构建邮件对象"""
        config = SYEmail._get_config()

        message = MIMEMultipart()
        message['From'] = Header(config.get('sender_name', ''), 'utf-8')
        message['To'] = Header(", ".join(to), 'utf-8')
        message['Subject'] = Header(subject, 'utf-8')
        message.attach(MIMEText(content, 'plain', 'utf-8'))

        if attachments:
            for file_path in attachments:
                if os.path.exists(file_path):
                    try:
                        with open(file_path, 'rb') as f:
                            # 使用 octet-stream 通用流类型
                            attachment = MIMEBase(
                                'application', 'octet-stream')
                            attachment.set_payload(f.read())
                            encoders.encode_base64(attachment)

                            # 解决文件名中文乱码问题
                            filename = os.path.basename(file_path)
                            attachment.add_header(
                                'Content-Disposition',
                                'attachment',
                                filename=('utf-8', '', filename)
                            )
                            message.attach(attachment)
                    except Exception as e:
                        SYLogger.error(f"⚠️ 读取附件失败 [{file_path}]: {e}")
                else:
                    SYLogger.error(f"⚠️ 附件文件不存在: {file_path}")
        return message

    @staticmethod
    def _send_smtp(message, to_list: List[str], verbose: bool) -> bool:
        """执行 SMTP 发送 (在线程池中运行)"""
        config = SYEmail._get_config()
        smtp_obj = None

        try:
            if verbose:
                SYLogger.info(
                    f"正在连接 {config.get('smtp_server')}:{config.get('smtp_port')} ...")

            # 1. 创建连接
            smtp_obj = smtplib.SMTP(
                config.get('smtp_server'),
                config.get('smtp_port'),
                timeout=10
            )

            if verbose:
                smtp_obj.set_debuglevel(1)

            # 2. 直接进行登录认证
            if verbose:
                SYLogger.info(f"正在使用账号 {config.get('smtp_user')} 登录...")

            smtp_obj.login(config.get('smtp_user'), config.get('smtp_pass'))

            if verbose:
                SYLogger.info("✅ 登录成功")

            # 3. 发送邮件
            smtp_obj.sendmail(
                config.get('sender_email'),
                to_list,
                message.as_string()
            )

            if verbose:
                SYLogger.info("✅ 邮件发送成功！")
            return True

        except smtplib.SMTPAuthenticationError:
            SYLogger.error("❌ 认证失败：请检查账号密码。")
        except smtplib.SMTPException as e:
            SYLogger.error(f"❌ SMTP 协议错误: {e}")
        except Exception as e:
            SYLogger.error(f"❌ 发送邮件时发生未知错误: {e}")
        finally:
            if smtp_obj:
                try:
                    smtp_obj.quit()
                except:
                    pass
        return False


# 使用示例
# async def main():
#     """异步主函数测试"""
#     print("开始发送邮件...")

#     success = await SYEmail.send(
#         to=["Osulcode.xiao@syholdings.com"],
#         subject="异步邮件测试",
#         content="这是一封通过异步方式发送的测试邮件。",
#         attachments=["test_file.txt"],
#         verbose=True
#     )

#     if success:
#         print("\n🎉 任务完成")
#     else:
#         print("\n⚠️ 任务失败")


# if __name__ == "__main__":
#     asyncio.run(main())
