"""Module containing extension app authenticator classes.
"""
from abc import ABC, abstractmethod

from symphony.bdk.core.auth.auth_session import AppAuthSession
from symphony.bdk.core.auth.exception import AuthInitializationError
from symphony.bdk.core.auth.jwt_helper import validate_jwt, create_signed_jwt
from symphony.bdk.core.auth.tokens_repository import TokensRepository, InMemoryTokensRepository
from symphony.bdk.core.config.model.bdk_rsa_key_config import BdkRsaKeyConfig
from symphony.bdk.gen import ApiException
from symphony.bdk.gen.login_api.authentication_api import AuthenticationApi
from symphony.bdk.gen.login_model.authenticate_extension_app_request import AuthenticateExtensionAppRequest
from symphony.bdk.gen.login_model.extension_app_tokens import ExtensionAppTokens
from symphony.bdk.gen.pod_api.pod_api import PodApi
from symphony.bdk.gen.pod_model.pod_certificate import PodCertificate


class ExtensionAppAuthenticator(ABC):
    """Base abstract class to handle extension app authentication.
    """

    @abstractmethod
    async def authenticate_extension_app(self, app_token: str) -> AppAuthSession:
        """Authenticates an extension app.

        :param app_token: the token generated by the application backend
        :return: the extension app authentication session
        """

    @abstractmethod
    async def validate_jwt(self, jwt: str) -> dict:
        """Validates a jwt against the pod certificate.

        :param jwt: the jwt to be validated
        :return: the dictionary of jwt claims
        :raise AuthInitializationError: If the pod certificate or jwt are invalid.
        """

    @abstractmethod
    async def is_token_pair_valid(self, app_token: str, symphony_token: str) -> bool:
        """Validates if appToken and symphonyToken corresponds to an existing session

        :param app_token: the application token
        :param symphony_token: the Symphony token
        :return: True if token pair is valid, False otherwise
        """

    @abstractmethod
    async def get_pod_certificate(self) -> PodCertificate:
        """Retrieves the pod certificate. Useful when validating JWT tokens.

        :return: the pod certificate
        """


class ExtensionAppAuthenticatorRsa(ExtensionAppAuthenticator):
    """A subclass of :class:`ExtensionAppAuthenticator` specific to RSA extension app authentication.
    """

    def __init__(self, authentication_api: AuthenticationApi, pod_api: PodApi, app_id: str,
                 private_key_config: BdkRsaKeyConfig, tokens_repository: TokensRepository = None):
        """

        :param login_client: the login ApiClient
        :param pod_client: the pod ApiClient
        :param app_id: the application ID
        :param private_key_config: the private key configuration of the extension app
        """
        self._authentication_api = authentication_api
        self._pod_api = pod_api
        self._app_id = app_id
        self._private_key_config = private_key_config
        self._tokens_repository = tokens_repository or InMemoryTokensRepository()

    async def authenticate_extension_app(self, app_token: str) -> AppAuthSession:
        auth_session = AppAuthSession(self, app_token)
        await auth_session.refresh()

        return auth_session

    async def authenticate_and_retrieve_tokens(self, app_token: str) -> ExtensionAppTokens:
        """Actually authenticates and retrieves the tokens.

        :param app_token: the token generated by the application backend
        :return: the extension app tokens
        """
        jwt = create_signed_jwt(self._private_key_config, self._app_id)
        authentication_request = AuthenticateExtensionAppRequest(app_token=app_token, auth_token=jwt)

        ext_app_tokens = await self._authentication_api.v1_pubkey_app_authenticate_extension_app_post(
            authentication_request)
        await self._tokens_repository.save(ext_app_tokens.app_token, ext_app_tokens.symphony_token)

        return ext_app_tokens

    async def validate_jwt(self, jwt: str) -> dict:
        try:
            pod_certificate = await self.get_pod_certificate()
        except ApiException as exc:
            raise AuthInitializationError(f"Unable to get the pod certificate: {exc.reason}") from exc

        return validate_jwt(jwt, pod_certificate.certificate, self._app_id)

    async def is_token_pair_valid(self, app_token: str, symphony_token: str) -> bool:
        retrieved_sym_token = await self._tokens_repository.get(app_token)
        return retrieved_sym_token is not None and retrieved_sym_token == symphony_token

    async def get_pod_certificate(self) -> PodCertificate:
        return await self._pod_api.v1_podcert_get()
