# Original Copyright 2021 OpenAI under MIT License.
# Modifications Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
# check_correctness_* functions are AWS additions

import contextlib
import faulthandler
import io
import multiprocessing
import os
import platform
import random
import shutil
import signal
import string
import subprocess
import tempfile
import time
import errno
from typing import Dict, Optional
import threading
lock = threading.Lock()


def check_correctness_java(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
    language="java",
    compile_timeout: float = 100,
):
    """
    Run all evaluation under java_exec_eval + randomized directory to avoid collision.
    Using subprocess with concurrent.futures for multi-thread evaluation.
    Make sure to clean up resources even if the test cases fail.
    """

    current_dir = os.path.dirname(os.path.realpath(__file__))
    entire_string = problem["prompt"] + completion + problem["test"]
    base_path = setup_base_path(current_dir, f"{language}_exec_eval", "")
    try:
        os.makedirs(base_path, exist_ok=False)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise
    path = os.path.join(base_path, f"main.{language}")

    with open(path, "w") as f:
        f.write(entire_string)

    try:
        exec_result_compile = subprocess.run(
            [f"javac", path],
            timeout=int(compile_timeout),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        compiled = exec_result_compile.returncode == 0
        if verbose:
            print("exec_result_compile", exec_result_compile)
        start = time.time()
        exec_result_run = subprocess.run(
            [f"java", "-cp", base_path, "Main"],
            timeout=int(timeout),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        elapsed = 1000.0 * (time.time() - start)
        if verbose:
            print("exec result run", exec_result_run)
        passed = exec_result_compile.returncode == 0 and exec_result_run.returncode == 0
        if exec_result_compile.returncode > 0:
            message = exec_result_compile.stderr
        else:
            message = exec_result_run.stderr

    except Exception as e:
        passed = False
        message = str(e)
        elapsed = None
        compiled = False

    try:
        shutil.rmtree(base_path)
    except Exception as e:
        if verbose:
            print(f"Error cleaning up directory {base_path}: {e}")

    return dict(
        task_id=problem["task_id"],
        passed=passed,
        result=message,
        completion_id=completion_id,
        time_elapsed=elapsed,
        compiled=compiled,
    )


def check_correctness_scala(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
    language="scala",
    compile_timeout: float = 100,
):

    current_dir = os.path.dirname(os.path.realpath(__file__))
    entire_string = problem["prompt"] + completion + problem["test"]
    base_path = setup_base_path(current_dir, f"{language}_exec_eval", "")
    try:
        os.makedirs(base_path, exist_ok=False)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise
    path = os.path.join(base_path, f"main.{language}")

    with open(path, "w") as f:
        f.write(entire_string)

    try:
        exec_result_compile = subprocess.run(
            [f"scalac", path, "-d", base_path],
            timeout=int(compile_timeout),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        compiled = exec_result_compile.returncode == 0
        if verbose:
            print("exec_result_compile", exec_result_compile)
        start = time.time()
        exec_result_run = subprocess.run(
            [f"scala", "-cp", base_path, "Main"],
            timeout=int(timeout),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        elapsed = 1000.0 * (time.time() - start)
        if verbose:
            print("exec result run", exec_result_run)
        passed = exec_result_compile.returncode == 0 and exec_result_run.returncode == 0
        if exec_result_compile.returncode > 0:
            message = exec_result_compile.stderr
        else:
            message = exec_result_run.stderr

    except Exception as e:
        passed = False
        message = str(e)
        elapsed = None
        compiled = False

    try:
        shutil.rmtree(base_path)
    except Exception as e:
        if verbose:
            print(f"Error cleaning up directory {base_path}: {e}")

    return dict(
        task_id=problem["task_id"],
        passed=passed,
        result=message,
        completion_id=completion_id,
        time_elapsed=elapsed,
        compiled=compiled,
    )


def check_correctness_perl(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="perl",
        extension=".pl",
        subprocess_command_lambda=lambda x: ["perl", f"{x}.pl"],
    )


def check_correctness_swift(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="swift",
        extension=".swift",
        subprocess_command_lambda=lambda x: ["swift", f"{x}.swift"],
    )


def check_correctness_javascript(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="javascript",
        extension=".js",
        subprocess_command_lambda=lambda x: ["node", f"{x}.js"],
    )


def check_correctness_typescript(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="typescript",
        extension=".ts",
        compile_command_lambda=lambda x: f"npx tsc {x} --target es5 --lib es2016".split(),
        subprocess_command_lambda=lambda x: ["node", f"{x}.js"],
    )


def check_correctness_ruby(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="ruby",
        extension=".rb",
        subprocess_command_lambda=lambda x: ["ruby", f"{x}.rb"],
    )


def check_correctness_kotlin(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="kotlin",
        extension=".kt",
        compile_command_lambda=lambda x: [
            "kotlinc",
            f"{x}.kt",
            "-include-runtime",
            "-d",
            f"{x}.jar",
        ],
        compile_timeout=100,  # needs longer than 20 sec
        subprocess_command_lambda=lambda x: ["java", "-jar", f"{x}.jar"],
        extra_cleanup=lambda x: f"{x}.jar",
    )


def check_correctness_php(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="php",
        extension=".php",
        subprocess_command_lambda=lambda x: ["php", f"{x}.php"],
    )


def check_correctness_go(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="go",
        extension=".go",
        subprocess_command_lambda=lambda x: ["go", "run", f"{x}.go"],
    )


def check_correctness_csharp(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
    compilation_timeout: float = 100,
):
    current_dir = os.path.dirname(os.path.realpath(__file__))
    program = problem["prompt"] + completion + problem["test"]
    # template c# project has all necessary DLLs
    template_cs_proj_zip = os.path.join(current_dir, "../resources/eval_csproj.zip")
    cs_eval_dir = setup_base_path(current_dir, "cs_eval", "")

    # extract zip into cs_eval_dir
    subprocess.check_call(
        f"unzip -q {template_cs_proj_zip} -d {cs_eval_dir}".split(), timeout=int(compilation_timeout)
    )

    passed, message = None, None
    compiled = False

    try:
        cs_project_path = os.path.join(cs_eval_dir, "eval_csproj")
        # entrypoint
        cs_program_path = os.path.join(cs_project_path, "Program.cs")
        with open(cs_program_path, "w") as f1:
            f1.write(program)
            f1.flush()

        compile_result = subprocess.run(
            f"dotnet build {cs_project_path}".split(),
            timeout=int(compilation_timeout),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        compiled = compile_result.returncode == 0
        message = compile_result.stderr

        if compiled:
            compiled_bin = os.path.join(cs_project_path, "bin/Debug/net6.0/eval_csproj")
            start = time.time()
            exec_result = subprocess.run(
                compiled_bin.split(),
                timeout=int(timeout),
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
            )
            passed = exec_result.returncode == 0
            message = exec_result.stderr
            elapsed = 1000.0 * (time.time() - start)
        else:
            passed, elapsed = False, None
    except Exception as e:
        if verbose:
            print(f"error occurred when running test cases: {e}")
        message = str(e)
        passed = False
        elapsed = None
    finally:
        try:
            shutil.rmtree(cs_eval_dir)
        except Exception as e:
            if verbose:
                print(f"Error trying to clean up directory: {e}")

    assert passed is not None, "should be either True or False"

    return dict(
        task_id=problem["task_id"],
        passed=passed,
        result=message,
        completion_id=completion_id,
        compiled=compiled,
        time_elapsed=elapsed,
    )


def check_correctness_cpp(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
):
    return check_correctness_helper(
        problem=problem,
        completion=completion,
        timeout=timeout,
        completion_id=completion_id,
        verbose=verbose,
        language="c#",
        extension=".cpp",
        compile_command_lambda=lambda x: [
            "g++",
            f"{os.path.basename(x)}.cpp",
            "-o",
            f"{os.path.basename(x)}_cpp",
        ],
        compile_timeout=100,
        subprocess_command_lambda=lambda x: [f"./{os.path.basename(x)}_cpp"],
        extra_cleanup=lambda x: f"{x}_cpp",
        cwd=True,
    )


def setup_base_path(
    current_dir,
    language_dirname,
    extension
):
    with lock:
        if not os.path.isdir(os.path.join(current_dir, language_dirname)):
            os.makedirs(os.path.join(current_dir, language_dirname))

    num_attempts, path = 0, None
    while True:
        num_attempts += 1
        if num_attempts > 10:
            assert False, "Unable to avoid filename collision"
        basename = "".join(
            random.choices(string.ascii_lowercase + string.ascii_uppercase, k=10)
        )

        base_path = os.path.join(current_dir, language_dirname, f"{basename}")
        path = base_path + f"{extension}"

        if extension == "":
            if not os.path.isdir(path):
                to_return = path
                break
        if not os.path.isfile(path):
            to_return = base_path
            break

    return to_return


def check_correctness_helper(
    problem: Dict,
    completion: str,
    timeout: float,
    completion_id: Optional[int] = None,
    verbose=False,
    language=None,
    extension=None,
    subprocess_command_lambda=None,
    compile_timeout=100,
    compile_command_lambda=None,
    extra_cleanup=None,
    cwd=None,
):
    current_dir = os.path.dirname(os.path.realpath(__file__))
    entire_string = problem["prompt"] + completion + problem["test"]

    language_dirname = f"{language}_exec_eval"

    base_path = setup_base_path(current_dir, language_dirname, extension)
    path = base_path + f"{extension}"

    if cwd is not None:
        cwd = os.path.dirname(base_path)
    with open(path, "w") as f:
        f.write(entire_string)
    try:
        if compile_command_lambda is not None:
            compile_result = subprocess.run(
                compile_command_lambda(base_path),
                timeout=int(compile_timeout),
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                cwd=cwd,
            )
            compiled = compile_result.returncode == 2 if language == "typescript" else compile_result.returncode == 0
        else:
            compiled = True

        if compiled:
            start = time.time()
            exec_result_run = subprocess.run(
                subprocess_command_lambda(base_path),
                timeout=int(timeout),
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
                cwd=cwd,
            )
            elapsed = 1000.0 * (time.time() - start)
            if verbose:
                print("exec result run", exec_result_run)

            passed = exec_result_run.returncode == 0
            message = exec_result_run.stderr
        else:
            passed, message, elapsed = False, compile_result.stderr, None

    except Exception as e:
        if verbose:
            print(f"error occurred when running test cases: {e}")
        message = str(e)
        passed, elapsed, compiled = False, None, False

    # clean up
    try:
        os.remove(path)
    except Exception as e:
        if verbose:
            print(f"Error trying to clean up file: {e}")
    try:
        if extra_cleanup is not None:
            extra_remove_path = extra_cleanup(base_path)
            assert isinstance(extra_remove_path, str)
            os.remove(extra_remove_path)
    except Exception as e:
        if verbose:
            print(f"Error trying to clean up file: {e}")

    # get result
    return dict(
        task_id=problem["task_id"],
        passed=passed,
        result=message,
        completion_id=completion_id,
        time_elapsed=elapsed,
        compiled=compiled,
    )


def check_correctness(
    problem: Dict, completion: str, timeout: float, completion_id: Optional[int] = None
) -> Dict:
    """
    Evaluates the functional correctness of a completion by running the test
    suite provided in the problem.
    :param completion_id: an optional completion ID so we can match
        the results later even if execution finishes asynchronously.
    """

    def unsafe_execute():

        with create_tempdir():

            # These system calls are needed when cleaning up tempdir.
            import os
            import shutil

            rmtree = shutil.rmtree
            rmdir = os.rmdir
            chdir = os.chdir

            # Disable functionalities that can make destructive changes to the test.
            reliability_guard()

            # Construct the check program and run it.
            check_program = (
                problem["prompt"]
                + completion
                + "\n"
                + problem["test"]
                + "\n"
                + f"check({problem['entry_point']})"
            )

            try:
                exec_globals = {}
                with swallow_io():
                    with time_limit(timeout):
                        # WARNING
                        # This program exists to execute untrusted model-generated code. Although
                        # it is highly unlikely that model-generated code will do something overtly
                        # malicious in response to this test suite, model-generated code may act
                        # destructively due to a lack of model capability or alignment.
                        # Users are strongly encouraged to sandbox this evaluation suite so that it
                        # does not perform destructive actions on their host or network. For more
                        # information on how OpenAI sandboxes its code, see the accompanying paper.
                        # Once you have read this disclaimer and taken appropriate precautions,
                        # uncomment the following line and proceed at your own risk:
                        exec(check_program, exec_globals)
                result.append("passed")
            except TimeoutException:
                result.append("timed out")
            except BaseException as e:
                result.append(f"failed: {e}")

            # Needed for cleaning up.
            shutil.rmtree = rmtree
            os.rmdir = rmdir
            os.chdir = chdir

    manager = multiprocessing.Manager()
    result = manager.list()

    start = time.time()
    p = multiprocessing.Process(target=unsafe_execute)
    p.start()
    p.join(timeout=timeout + 1)
    if p.is_alive():
        p.kill()
    elapsed = 1000.0 * (time.time() - start)

    if not result:
        result.append("timed out")

    return dict(
        task_id=problem["task_id"],
        passed=result[0] == "passed",
        result=result[0],
        completion_id=completion_id,
        time_elapsed=elapsed,
    )


@contextlib.contextmanager
def time_limit(seconds: float):
    def signal_handler(signum, frame):
        raise TimeoutException("Timed out!")

    signal.setitimer(signal.ITIMER_REAL, seconds)
    signal.signal(signal.SIGALRM, signal_handler)
    try:
        yield
    finally:
        signal.setitimer(signal.ITIMER_REAL, 0)


@contextlib.contextmanager
def swallow_io():
    stream = WriteOnlyStringIO()
    with contextlib.redirect_stdout(stream):
        with contextlib.redirect_stderr(stream):
            with redirect_stdin(stream):
                yield


@contextlib.contextmanager
def create_tempdir():
    with tempfile.TemporaryDirectory() as dirname:
        with chdir(dirname):
            yield dirname


class TimeoutException(Exception):
    pass


class WriteOnlyStringIO(io.StringIO):
    """StringIO that throws an exception when it's read from"""

    def read(self, *args, **kwargs):
        raise IOError

    def readline(self, *args, **kwargs):
        raise IOError

    def readlines(self, *args, **kwargs):
        raise IOError

    def readable(self, *args, **kwargs):
        """Returns True if the IO object can be read."""
        return False


class redirect_stdin(contextlib._RedirectStream):  # type: ignore
    _stream = "stdin"


@contextlib.contextmanager
def chdir(root):
    if root == ".":
        yield
        return
    cwd = os.getcwd()
    os.chdir(root)
    try:
        yield
    except BaseException as exc:
        raise exc
    finally:
        os.chdir(cwd)


def reliability_guard(maximum_memory_bytes: Optional[int] = None):
    """
    This disables various destructive functions and prevents the generated code
    from interfering with the test (e.g. fork bomb, killing other processes,
    removing filesystem files, etc.)
    WARNING
    This function is NOT a security sandbox. Untrusted code, including, model-
    generated code, should not be blindly executed outside of one. See the
    Codex paper for more information about OpenAI's code sandbox, and proceed
    with caution.
    """

    if maximum_memory_bytes is not None:
        import resource

        resource.setrlimit(
            resource.RLIMIT_AS, (maximum_memory_bytes, maximum_memory_bytes)
        )
        resource.setrlimit(
            resource.RLIMIT_DATA, (maximum_memory_bytes, maximum_memory_bytes)
        )
        if not platform.uname().system == "Darwin":
            resource.setrlimit(
                resource.RLIMIT_STACK, (maximum_memory_bytes, maximum_memory_bytes)
            )

    faulthandler.disable()

    import builtins

    builtins.exit = None
    builtins.quit = None

    import os

    os.environ["OMP_NUM_THREADS"] = "1"

    os.kill = None
    os.system = None
    os.putenv = None
    os.remove = None
    os.removedirs = None
    os.rmdir = None
    os.fchdir = None
    os.setuid = None
    os.fork = None
    os.forkpty = None
    os.killpg = None
    os.rename = None
    os.renames = None
    os.truncate = None
    os.replace = None
    os.unlink = None
    os.fchmod = None
    os.fchown = None
    os.chmod = None
    os.chown = None
    os.chroot = None
    os.fchdir = None
    os.lchflags = None
    os.lchmod = None
    os.lchown = None
    os.getcwd = None
    os.chdir = None

    import shutil

    shutil.rmtree = None
    shutil.move = None
    shutil.chown = None

    import subprocess

    subprocess.Popen = None  # type: ignore

    __builtins__["help"] = None

    import sys

    sys.modules["ipdb"] = None
    sys.modules["joblib"] = None
    sys.modules["resource"] = None
    sys.modules["psutil"] = None
    sys.modules["tkinter"] = None
