"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.proxyHttpRequest = proxyHttpRequest;
const http = require("http");
const https = require("https");
const socks_proxy_agent_1 = require("socks-proxy-agent");
const index_1 = require("./index");
async function proxyHttpRequest(target, isHttps, request) {
    async function requestPromise() {
        (0, index_1.logDebug)('proxyHttpRequest', JSON.stringify({ target, isHttps, request }, null, 2));
        const socksProxyAgent = new socks_proxy_agent_1.SocksProxyAgent('socks://localhost:1055');
        return new Promise((resolve, reject) => {
            const chunks = [];
            const httpLib = isHttps == undefined ?
                (target.port == 443 ? https : http) :
                (isHttps ? https : http);
            const apiRequest = httpLib.request({
                ...target,
                agent: socksProxyAgent,
                path: request.path,
                method: request.method,
                headers: request.headers,
            }, (res) => {
                res.on('data', (chunk) => {
                    chunks.push(chunk);
                });
                res.on('end', () => {
                    const responseBody = Buffer.concat(chunks);
                    (0, index_1.logDebug)('requestPromise.end responseBody', responseBody);
                    resolve({
                        statusCode: res.statusCode || 500,
                        headers: res.headers,
                        body: responseBody.toString('base64'),
                        isBase64Encoded: true,
                    });
                });
                res.on('error', (error) => {
                    (0, index_1.logError)('Error receiving response:', error);
                    reject(error);
                });
            });
            apiRequest.on('error', (error) => {
                (0, index_1.logError)('Error sending request:', error);
                reject(error);
            });
            if (request.body != null) {
                apiRequest.write(request.body);
            }
            apiRequest.end();
        });
    }
    const connectionRetryDelays = [10, 50, 100, 500, 1000, 2000, 3000];
    let attempt = 0;
    let success = false;
    let response;
    do {
        try {
            response = await requestPromise();
            success = true;
        }
        catch (error) {
            if (error == 'Error: Socks5 proxy rejected connection - Failure' && attempt < connectionRetryDelays.length) {
                (0, index_1.logError)('Error: Socks5 proxy rejected connection - Failure');
                (0, index_1.logInfo)('Retrying in', connectionRetryDelays[attempt], 'ms');
                await new Promise((resolve) => setTimeout(resolve, connectionRetryDelays[attempt]));
                attempt++;
            }
            else {
                throw error;
            }
        }
    } while (!success && attempt < connectionRetryDelays.length);
    if (attempt > 0) {
        (0, index_1.logInfo)('Error: Socks5 proxy rejected connection - Failure - RESOLVED - attempt:', attempt, 'total delay time:', connectionRetryDelays.slice(0, attempt).reduce((a, b) => a + b, 0));
    }
    return response;
}
//# sourceMappingURL=data:application/json;base64,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