"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = handler;
const http = require("http");
const metrics_1 = require("@aws-lambda-powertools/metrics");
const socks_proxy_agent_1 = require("socks-proxy-agent");
async function proxyHttpRequest(target, request) {
    return new Promise((resolve, reject) => {
        const chunks = [];
        const apiRequest = http.request({
            ...target,
            path: request.path,
            method: request.method,
            headers: request.headers,
        }, (res) => {
            res.on('data', (chunk) => {
                chunks.push(chunk);
            });
            res.on('end', () => {
                const responseBody = Buffer.concat(chunks);
                resolve({
                    statusCode: res.statusCode || 500,
                    headers: res.headers,
                    body: responseBody.toString('base64'),
                    isBase64Encoded: true,
                });
            });
            res.on('error', (error) => {
                console.error('Error receiving response:', error);
                reject(error);
            });
        });
        apiRequest.on('error', (error) => {
            console.error('Error sending request:', error);
            reject(error);
        });
        if (request.body != null) {
            apiRequest.write(request.body);
        }
        apiRequest.end();
    });
}
async function handler(event) {
    let metrics;
    try {
        const socksProxyAgent = new socks_proxy_agent_1.SocksProxyAgent('socks://localhost:1055');
        if (!event.headers['ts-target-ip']) {
            return {
                statusCode: 400,
                headers: {
                    'ts-error': "Missing header 'ts-target-ip'",
                },
            };
        }
        if (!event.headers['ts-target-port']) {
            return {
                statusCode: 400,
                headers: {
                    'ts-error': "Missing header 'ts-target-port'",
                },
            };
        }
        if (event.headers['ts-metric-service']) {
            metrics = new metrics_1.Metrics({ namespace: 'tailscale-service', serviceName: event.headers['ts-metric-service'] });
            if (event.headers['ts-metric-dimension-name'] && event.headers['ts-metric-dimension-value']) {
                metrics.addDimensions({
                    [event.headers['ts-metric-dimension-name']]: event.headers['ts-metric-dimension-value'],
                });
            }
        }
        const targetHeaders = { ...event.headers };
        delete targetHeaders['ts-target-ip'];
        delete targetHeaders['ts-target-port'];
        if (targetHeaders['ts-metric-service']) {
            delete targetHeaders['ts-metric-service'];
        }
        if (targetHeaders['ts-metric-dimension-name']) {
            delete targetHeaders['ts-metric-dimension-name'];
        }
        if (targetHeaders['ts-metric-dimension-value']) {
            delete targetHeaders['ts-metric-dimension-value'];
        }
        const response = await proxyHttpRequest({
            hostname: event.headers['ts-target-ip'],
            port: event.headers['ts-target-port'],
            agent: socksProxyAgent,
        }, {
            path: event.requestContext.http.path,
            headers: targetHeaders,
            method: event.requestContext.http.method,
            body: event.body,
        });
        metrics?.addMetric('success', metrics_1.MetricUnit.Count, 1);
        return response;
    }
    catch (_err) {
        metrics?.addMetric('error', metrics_1.MetricUnit.Count, 1);
        const err = _err;
        return {
            statusCode: 500,
            headers: {
                'ts-error-name': err.name,
                'ts-error-message': err.message,
            },
        };
    }
    finally {
        metrics?.publishStoredMetrics();
    }
}
//# sourceMappingURL=data:application/json;base64,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