"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnvironmentVariablesService = void 0;
/**
 * This class is used to fetch environment variables that are available in the execution environment.
 *
 * These variables can be a mix of runtime environment variables set by AWS and
 * other environment variables that are set by the developer to configure Powertools for AWS Lambda.
 *
 * @example
 * ```typescript
 * import { EnvironmentVariablesService } from '@aws-lambda-powertools/commons/';
 *
 * const config = new EnvironmentVariablesService();
 * const serviceName = config.getServiceName();
 * ```
 *
 * @see https://docs.powertools.aws.dev/lambda/typescript/latest/#environment-variables
 *
 * @class
 */
class EnvironmentVariablesService {
    /**
     * Increase JSON indentation for Logger to ease debugging when running functions locally or in a non-production environment
     */
    devModeVariable = 'POWERTOOLS_DEV';
    /**
     * Set service name used for tracing namespace, metrics dimension and structured logging
     */
    serviceNameVariable = 'POWERTOOLS_SERVICE_NAME';
    /**
     * AWS X-Ray Trace ID environment variable
     * @private
     */
    xRayTraceIdVariable = '_X_AMZN_TRACE_ID';
    /**
     * Get the value of an environment variable by name.
     *
     * @param {string} name The name of the environment variable to fetch.
     */
    get(name) {
        return process.env[name]?.trim() || '';
    }
    /**
     * Get the value of the `POWERTOOLS_SERVICE_NAME` environment variable.
     */
    getServiceName() {
        return this.get(this.serviceNameVariable);
    }
    /**
     * Get the value of the `_X_AMZN_TRACE_ID` environment variable.
     *
     * The AWS X-Ray Trace data available in the environment variable has this format:
     * `Root=1-5759e988-bd862e3fe1be46a994272793;Parent=557abcec3ee5a047;Sampled=1`,
     *
     * The actual Trace ID is: `1-5759e988-bd862e3fe1be46a994272793`.
     */
    getXrayTraceId() {
        const xRayTraceData = this.getXrayTraceData();
        return xRayTraceData?.Root;
    }
    /**
     * Determine if the current invocation is part of a sampled X-Ray trace.
     *
     * The AWS X-Ray Trace data available in the environment variable has this format:
     * `Root=1-5759e988-bd862e3fe1be46a994272793;Parent=557abcec3ee5a047;Sampled=1`,
     */
    getXrayTraceSampled() {
        const xRayTraceData = this.getXrayTraceData();
        return xRayTraceData?.Sampled === '1';
    }
    /**
     * Determine if the current invocation is running in a development environment.
     */
    isDevMode() {
        return this.isValueTrue(this.get(this.devModeVariable));
    }
    /**
     * Helper function to determine if a value is considered thruthy.
     *
     * @param value The value to check for truthiness.
     */
    isValueTrue(value) {
        const truthyValues = ['1', 'y', 'yes', 't', 'true', 'on'];
        return truthyValues.includes(value.toLowerCase());
    }
    /**
     * Helper function to determine if a value is considered falsy.
     *
     * @param value The value to check for falsiness.
     */
    isValueFalse(value) {
        const falsyValues = ['0', 'n', 'no', 'f', 'false', 'off'];
        return falsyValues.includes(value.toLowerCase());
    }
    /**
     * Get the AWS X-Ray Trace data from the environment variable.
     *
     * The method parses the environment variable `_X_AMZN_TRACE_ID` and returns an object with the key-value pairs.
     */
    getXrayTraceData() {
        const xRayTraceEnv = this.get(this.xRayTraceIdVariable);
        if (xRayTraceEnv === '')
            return undefined;
        if (!xRayTraceEnv.includes('='))
            return { Root: xRayTraceEnv };
        const xRayTraceData = {};
        for (const field of xRayTraceEnv.split(';')) {
            const [key, value] = field.split('=');
            xRayTraceData[key] = value;
        }
        return xRayTraceData;
    }
}
exports.EnvironmentVariablesService = EnvironmentVariablesService;
