"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isSdkClient = exports.addUserAgentMiddleware = exports.customUserAgentMiddleware = void 0;
const version_js_1 = require("./version.js");
const EXEC_ENV = process.env.AWS_EXECUTION_ENV || 'NA';
const middlewareOptions = {
    relation: 'after',
    toMiddleware: 'getUserAgentMiddleware',
    name: 'addPowertoolsToUserAgent',
    tags: ['POWERTOOLS', 'USER_AGENT'],
};
/**
 * Type guard to check if the client provided is a valid AWS SDK v3 client.
 *
 * @internal
 */
const isSdkClient = (client) => typeof client === 'object' &&
    client !== null &&
    'send' in client &&
    typeof client.send === 'function' &&
    'config' in client &&
    client.config !== undefined &&
    typeof client.config === 'object' &&
    client.config !== null &&
    'middlewareStack' in client &&
    client.middlewareStack !== undefined &&
    typeof client.middlewareStack === 'object' &&
    client.middlewareStack !== null &&
    'identify' in client.middlewareStack &&
    typeof client.middlewareStack.identify === 'function' &&
    'addRelativeTo' in client.middlewareStack &&
    typeof client.middlewareStack.addRelativeTo === 'function';
exports.isSdkClient = isSdkClient;
/**
 * Helper function to create a custom user agent middleware for the AWS SDK v3 clients.
 *
 * The middleware will append the provided feature name and the current version of
 * the Powertools for AWS Lambda library to the user agent string.
 *
 * @example "PT/Tracer/2.1.0 PTEnv/nodejs20x"
 *
 * @param feature The feature name to be added to the user agent
 *
 * @internal
 */
const customUserAgentMiddleware = (feature) => {
    return (next) => async (args) => {
        const powertoolsUserAgent = `PT/${feature}/${version_js_1.PT_VERSION} PTEnv/${EXEC_ENV}`;
        args.request.headers['user-agent'] =
            `${args.request.headers['user-agent']} ${powertoolsUserAgent}`;
        return await next(args);
    };
};
exports.customUserAgentMiddleware = customUserAgentMiddleware;
/**
 * Check if the provided middleware stack already has the Powertools for AWS Lambda
 * user agent middleware.
 *
 * @param middlewareStack The middleware stack to check
 *
 * @internal
 */
const hasPowertools = (middlewareStack) => {
    let found = false;
    for (const middleware of middlewareStack) {
        if (middleware.includes('addPowertoolsToUserAgent')) {
            found = true;
        }
    }
    return found;
};
/**
 * Add the Powertools for AWS Lambda user agent middleware to the
 * AWS SDK v3 client provided.
 *
 * We use this middleware to unbotrusively track the usage of the library
 * and secure continued investment in the project.
 *
 * @param client The AWS SDK v3 client to add the middleware to
 * @param feature The feature name to be added to the user agent
 */
const addUserAgentMiddleware = (client, feature) => {
    try {
        if (isSdkClient(client)) {
            if (hasPowertools(client.middlewareStack.identify())) {
                return;
            }
            client.middlewareStack.addRelativeTo(customUserAgentMiddleware(feature), middlewareOptions);
        }
        else {
            throw new Error('The client provided does not match the expected interface');
        }
    }
    catch (error) {
        console.warn('Failed to add user agent middleware', error);
    }
};
exports.addUserAgentMiddleware = addUserAgentMiddleware;
