#!/usr/bin/env python3
"""Generate explicit schemas from API reference"""

import json
from datetime import date, datetime
from decimal import Decimal
from pathlib import Path

# Type mapping from API types to Python types
TYPE_MAP = {
    "string": str,
    "integer": int,
    "number": Decimal,
    "boolean": bool,
    "date": date,
    "date-time": datetime,
    "object": dict,
}


def resolve_ref(ref_path: str, components: dict) -> dict:
    """Resolve a $ref to the actual schema"""
    parts = ref_path.split("/")
    if parts[0] == "#" and parts[1] == "components" and parts[2] == "schemas":
        model_name = parts[3]
        return components.get(model_name, {})
    return {}


def convert_api_type_to_python(
    api_field: dict, components: dict
) -> tuple[type, bool, bool, str | None]:
    """Convert API field type to Python type info

    Returns: (type, is_optional, is_list, nested_model_name)
    """
    is_optional = api_field.get("nullable", False)
    is_list = False
    nested_model = None
    base_type = str

    if "$ref" in api_field:
        ref = api_field["$ref"]
        nested_model = ref.split("/")[-1]
        base_type = dict
    elif "type" in api_field:
        api_type = api_field["type"]
        if api_type == "array":
            is_list = True
            items = api_field.get("items", {})
            if "$ref" in items:
                ref = items["$ref"]
                nested_model = ref.split("/")[-1]
                base_type = dict
            elif "type" in items:
                base_type = TYPE_MAP.get(items["type"], str)
        else:
            # Check format for date/datetime
            if api_type == "string":
                format_type = api_field.get("format", "")
                if format_type == "date":
                    base_type = date
                elif format_type == "date-time":
                    base_type = datetime
                else:
                    base_type = TYPE_MAP.get(api_type, str)
            else:
                base_type = TYPE_MAP.get(api_type, str)
    else:
        # Unknown type, default to dict
        base_type = dict

    return base_type, is_optional, is_list, nested_model


def generate_field_schema(
    field_name: str, api_field: dict, components: dict, required_fields: set
) -> str:
    """Generate FieldSchema line for a field"""
    is_required = field_name in required_fields
    base_type, is_optional, is_list, nested_model = convert_api_type_to_python(
        api_field, components
    )

    # Override optionality based on required fields
    if is_required:
        is_optional = False

    type_name = base_type.__name__ if hasattr(base_type, "__name__") else str(base_type)

    nested_str = ""
    if nested_model:
        nested_str = f', nested_model="{nested_model}"'

    return f'    "{field_name}": FieldSchema(name="{field_name}", type={type_name}, is_optional={is_optional}, is_list={is_list}{nested_str}),'


def generate_model_schema(model_name: str, api_schema: dict, components: dict) -> str:
    """Generate schema definition for a model"""
    properties = api_schema.get("properties", {})
    required = set(api_schema.get("required", []))

    lines = [f"{model_name.upper()}_SCHEMA: dict[str, FieldSchema] = {{"]

    for field_name in sorted(properties.keys()):
        api_field = properties[field_name]
        line = generate_field_schema(field_name, api_field, components, required)
        lines.append(line)

    lines.append("}")
    return "\n".join(lines)


# Load API schema
with open("api_schema_full.json") as f:
    api_schema = json.load(f)

components = api_schema.get("components", {}).get("schemas", {})

# Models to generate
models_to_generate = {
    "PremiumContractDetail": "CONTRACT",
    "EntityDetail": "ENTITY",
    "GrantOpportunityDetail": "GRANT",
    "Forecast": "FORECAST",
    "OpportunityDetail": "OPPORTUNITY",
    "NoticeDetail": "NOTICE",
    "Agency": "AGENCY",
    "PlaceOfPerformance": "LOCATION",
    "Recipient": "RECIPIENT_PROFILE",
}

print("# Generated schemas from API reference")
print("# DO NOT EDIT MANUALLY - Generated by generate_schemas_from_api.py")
print()

for api_model_name, our_schema_name in models_to_generate.items():
    if api_model_name not in components:
        print(f"# WARNING: {api_model_name} not found in API schema")
        continue

    api_model = components[api_model_name]
    schema_code = generate_model_schema(our_schema_name, api_model, components)
    print(schema_code)
    print()
