import asyncio
import logging
from typing import Any, Callable, Optional, TYPE_CHECKING

if TYPE_CHECKING:
    from ..API import GreenApi

class Webhooks:
    _running: Optional[bool] = None

    def __init__(self, api: "GreenApi"):
        self.api = api

    def startReceivingNotifications(
            self, onEvent: Callable[[str, dict], Any]
    ) -> None:
        self._running = True

        self._start_polling(onEvent)

    async def startReceivingNotificationsAsync(
            self, onEvent: Callable[[str, dict], Any]
    ) -> None:
        self._running = True

        await self._start_pollingAsync(onEvent)

    def stopReceivingNotifications(self) -> None:
        self._running = False

    async def stopReceivingNotificationsAsync(self) -> None:
        self._running = False

    def _start_polling(self, handler: Callable[[str, dict], Any]) -> None:
        self.api.session.headers["Connection"] = "keep-alive"

        self.api.logger.log(
            logging.INFO, "Started receiving incoming notifications."
        )

        while self._running:
            try:
                response = self.api.receiving.receiveNotification()
                if response.code == 200:
                    if not response.data:
                        continue
                    response = response.data

                    body = response["body"]
                    type_webhook = body["typeWebhook"]

                    handler(type_webhook, body)

                    self.api.receiving.deleteNotification(
                        response["receiptId"]
                    )
            except KeyboardInterrupt:
                break

        self.api.session.headers["Connection"] = "close"

        self.api.logger.log(
            logging.INFO, "Stopped receiving incoming notifications."
        )

    async def _start_pollingAsync(self, handler: Callable[[str, dict], Any]) -> None:
        self.api.session.headers["Connection"] = "keep-alive"

        self.api.logger.log(
            logging.INFO, "Started receiving incoming notifications asynchronously."
        )

        while self._running:
            try:
                response = await self.api.receiving.receiveNotificationAsync()
                if response.code == 200:
                    if not response.data:
                        await asyncio.sleep(0.1)
                        continue
                    response = response.data

                    body = response["body"]
                    type_webhook = body["typeWebhook"]

                    handler(type_webhook, body)

                    await self.api.receiving.deleteNotificationAsync(
                        response["receiptId"]
                    )
            except KeyboardInterrupt:
                break

        self.api.session.headers["Connection"] = "close"

        self.api.logger.log(
            logging.INFO, "Stopped receiving incoming notifications asynchronously."
        )