"""TelemetryFlow integration module.

TelemetryFlow Python SDK - Community Enterprise Observability Platform (CEOP)
Copyright (c) 2024-2026 DevOpsCorner Indonesia. All rights reserved.

Compatible with TFO-Collector v${tfo_collector_version} (OCB-native)
SDK Version: ${sdk_version}

Auto-generated by telemetryflow-gen.
"""

from __future__ import annotations

import atexit
import os
from typing import TYPE_CHECKING

from telemetryflow import TelemetryFlowBuilder
from telemetryflow.application.commands import SpanKind

if TYPE_CHECKING:
    from telemetryflow.client import TelemetryFlowClient

# Global client instance
_client: TelemetryFlowClient | None = None


def init(
    use_v2_api: bool = True,
    v2_only: bool = False,
    collector_name: str | None = None,
) -> TelemetryFlowClient:
    """Initialize TelemetryFlow SDK with TFO v2 API support.

    Args:
        use_v2_api: Enable TFO v2 API endpoints (default: True)
        v2_only: Enable v2-only mode, disabling v1 endpoints (default: False)
        collector_name: Custom collector name for identity (optional)

    Returns:
        TelemetryFlowClient: Initialized client instance.

    Example:
        >>> from telemetry import init, get_client
        >>> init()  # Uses v2 API by default
        >>> client = get_client()
        >>> client.log_info("Application started")

        >>> # Or with v2-only mode
        >>> init(v2_only=True)
    """
    global _client

    if _client is not None:
        return _client

    builder = TelemetryFlowBuilder().with_auto_configuration()

    # Apply TFO v2 API settings if the builder supports them
    if hasattr(builder, "with_v2_api"):
        builder = builder.with_v2_api(use_v2_api)
    if v2_only and hasattr(builder, "with_v2_only"):
        builder = builder.with_v2_only()
    if collector_name and hasattr(builder, "with_collector_name"):
        builder = builder.with_collector_name(collector_name)

    _client = builder.build()
    _client.initialize()

    # Register shutdown on exit
    atexit.register(shutdown)

    return _client


def init_v2_only(collector_name: str | None = None) -> TelemetryFlowClient:
    """Initialize TelemetryFlow SDK in v2-only mode.

    This is a convenience function for production deployments
    that only use TFO v2 API endpoints.

    Args:
        collector_name: Custom collector name for identity (optional)

    Returns:
        TelemetryFlowClient: Initialized client instance.

    Example:
        >>> from telemetry import init_v2_only, get_client
        >>> init_v2_only()
        >>> client = get_client()
        >>> client.log_info("Application started with v2-only mode")
    """
    return init(use_v2_api=True, v2_only=True, collector_name=collector_name)


def get_client() -> TelemetryFlowClient:
    """Get the global TelemetryFlow client.

    Returns:
        TelemetryFlowClient: The initialized client.

    Raises:
        RuntimeError: If SDK is not initialized.
    """
    if _client is None:
        raise RuntimeError("TelemetryFlow SDK not initialized. Call init() first.")
    return _client


def shutdown() -> None:
    """Shutdown TelemetryFlow SDK gracefully."""
    global _client
    if _client is not None:
        _client.shutdown()
        _client = None


# Convenience re-exports
__all__ = [
    "init",
    "init_v2_only",
    "get_client",
    "shutdown",
    "SpanKind",
]
