# ${project_name} Development Guide

## Prerequisites

- Python 3.12+
- PostgreSQL (or SQLite for development)
- Docker & Docker Compose (optional)

## Setup

### 1. Clone and Install

```bash
cd ${project_name}

# Create virtual environment
python -m venv .venv
source .venv/bin/activate  # On Windows: .venv\Scripts\activate

# Install dependencies
pip install -e ".[dev]"
```

### 2. Configure Environment

```bash
# Copy example configuration
cp .env.example .env

# Edit with your settings
vim .env
```

### 3. Start Database

**Option A: Docker**

```bash
make docker-up
```

**Option B: Local PostgreSQL**

```bash
# Create database
createdb ${db_name}
```

### 4. Run Migrations

```bash
make migrate
```

### 5. Start Development Server

```bash
make run
```

The server will be available at `http://localhost:${server_port}`

## Development Workflow

### Running Tests

```bash
# Run all tests
make test

# Run with coverage
pytest tests/ -v --cov=src/${module_name}

# Run specific test file
pytest tests/unit/test_user.py -v
```

### Code Quality

```bash
# Run linting
make lint

# Format code
make format

# Type checking
mypy src/
```

### Database Migrations

```bash
# Create new migration
make migrate-new

# Apply migrations
make migrate

# Rollback migration
alembic downgrade -1
```

## Adding New Entities

Use the TelemetryFlow generator:

```bash
telemetryflow-restapi entity -n Product -f 'name:string,price:float,description:text?,active:bool'
```

This creates:

| File | Description |
|------|-------------|
| `domain/entity/product.py` | Domain entity |
| `domain/repository/product_repository.py` | Repository interface |
| `application/command/product_commands.py` | CQRS commands |
| `application/query/product_queries.py` | CQRS queries |
| `application/handler/product_*_handler.py` | Command/Query handlers |
| `application/dto/product_dto.py` | Data transfer objects |
| `infrastructure/persistence/product_repository.py` | SQLAlchemy repository |
| `infrastructure/http/handlers/product_handler.py` | HTTP endpoints |
| `migrations/*_create_products.sql` | Database migration |

### Register the Routes

Edit `infrastructure/http/routes.py`:

```python
from ${module_name}.infrastructure.http.handlers.product_handler import product_bp

def register_routes(app: Flask) -> None:
    # ... existing routes ...
    app.register_blueprint(product_bp, url_prefix="/api/v1")
```

### Run Migration

```bash
make migrate
```

## Environment Variables

| Variable | Description | Default |
|----------|-------------|---------|
| `${env_prefix}_HOST` | Server host | `0.0.0.0` |
| `${env_prefix}_PORT` | Server port | `${server_port}` |
| `${env_prefix}_DEBUG` | Debug mode | `false` |
| `${env_prefix}_DB_DRIVER` | Database driver | `${db_driver}` |
| `${env_prefix}_DB_HOST` | Database host | `${db_host}` |
| `${env_prefix}_DB_PORT` | Database port | `${db_port}` |
| `${env_prefix}_DB_NAME` | Database name | `${db_name}` |
| `${env_prefix}_DB_USER` | Database user | `${db_user}` |
| `${env_prefix}_DB_PASSWORD` | Database password | - |
| `${env_prefix}_SECRET_KEY` | Flask secret key | - |
| `${env_prefix}_JWT_SECRET_KEY` | JWT secret key | - |

## Docker Development

### Build Image

```bash
make docker-build
```

### Run with Docker Compose

```bash
make docker-up
```

### View Logs

```bash
docker compose logs -f app
```

### Stop Services

```bash
make docker-down
```

## Debugging

### VS Code Configuration

Create `.vscode/launch.json`:

```json
{
  "version": "0.2.0",
  "configurations": [
    {
      "name": "Flask",
      "type": "python",
      "request": "launch",
      "module": "${module_name}.main",
      "env": {
        "${env_prefix}_DEBUG": "true"
      },
      "justMyCode": false
    }
  ]
}
```

### Enable Debug Mode

Set `${env_prefix}_DEBUG=true` in your `.env` file.

## Observability

### TelemetryFlow Integration

The application includes TelemetryFlow for observability:

```python
from telemetryflow import TelemetryFlowBuilder

client = TelemetryFlowBuilder().with_auto_configuration().build()
client.initialize()
```

Configure TelemetryFlow in `.env`:

```bash
TELEMETRYFLOW_API_KEY_ID=your_key_id
TELEMETRYFLOW_API_KEY_SECRET=your_key_secret
TELEMETRYFLOW_SERVICE_NAME=${service_name}
```

## Troubleshooting

### Database Connection Issues

1. Verify database is running
2. Check connection string in `.env`
3. Ensure database exists: `createdb ${db_name}`

### Migration Errors

```bash
# Reset migrations (development only!)
alembic downgrade base
alembic upgrade head
```

### Port Already in Use

```bash
# Find process using port
lsof -i :${server_port}

# Kill process
kill -9 <PID>
```

---

*Generated by TelemetryFlow RESTful API Generator*
