"""gRPC server with TelemetryFlow instrumentation example.

TelemetryFlow Python SDK - Community Enterprise Observability Platform (CEOP)
Copyright (c) 2024-2026 DevOpsCorner Indonesia. All rights reserved.

Auto-generated by telemetryflow-gen.

Note: This example requires grpcio. Install with:
    pip install grpcio grpcio-tools
"""

import time
from concurrent import futures

try:
    import grpc
except ImportError:
    print("Please install grpcio: pip install grpcio grpcio-tools")
    raise

from telemetry import init, get_client, SpanKind


class TelemetryInterceptor(grpc.ServerInterceptor):
    """gRPC server interceptor for telemetry."""

    def intercept_service(self, continuation, handler_call_details):
        """Intercept and instrument gRPC calls."""
        client = get_client()
        method = handler_call_details.method
        start_time = time.time()

        # Start span
        span_id = client.start_span(
            f"gRPC {method}",
            SpanKind.SERVER,
            {
                "rpc.system": "grpc",
                "rpc.method": method,
            },
        )

        # Increment counter
        client.increment_counter(
            "grpc.requests.total",
            attributes={"method": method},
        )

        try:
            response = continuation(handler_call_details)
            client.add_span_event(span_id, "response_sent")
            return response
        except Exception as e:
            client.log_error(f"gRPC call failed: {e}")
            client.increment_counter(
                "grpc.errors.total",
                attributes={"method": method},
            )
            client.end_span(span_id, e)
            raise
        finally:
            duration = time.time() - start_time
            client.record_histogram(
                "grpc.request.duration",
                duration,
                unit="s",
                attributes={"method": method},
            )
            client.end_span(span_id)


def main() -> None:
    """Start the gRPC server with telemetry."""
    # Initialize TelemetryFlow SDK
    init()
    client = get_client()
    client.log_info("Starting gRPC server with TelemetryFlow instrumentation")

    # Create server with interceptor
    server = grpc.server(
        futures.ThreadPoolExecutor(max_workers=10),
        interceptors=[TelemetryInterceptor()],
    )

    # Add your service implementations here
    # example_pb2_grpc.add_ExampleServicer_to_server(ExampleServicer(), server)

    server.add_insecure_port("[::]:50051")
    server.start()

    print("gRPC Server running on port 50051")
    print("Press Ctrl+C to stop")

    try:
        server.wait_for_termination()
    except KeyboardInterrupt:
        print("\nShutting down...")
        server.stop(grace=5)
        print("Server stopped.")


if __name__ == "__main__":
    main()
