"""HTTP server with TelemetryFlow instrumentation example.

TelemetryFlow Python SDK - Community Enterprise Observability Platform (CEOP)
Copyright (c) 2024-2026 DevOpsCorner Indonesia. All rights reserved.

Auto-generated by telemetryflow-gen.
"""

import time
from http.server import HTTPServer, BaseHTTPRequestHandler

from telemetry import init, get_client, SpanKind


class InstrumentedHandler(BaseHTTPRequestHandler):
    """HTTP handler with telemetry instrumentation."""

    def do_GET(self) -> None:
        """Handle GET requests with instrumentation."""
        client = get_client()
        start_time = time.time()

        # Start span for the request
        span_id = client.start_span(
            f"HTTP GET {self.path}",
            SpanKind.SERVER,
            {
                "http.method": "GET",
                "http.url": self.path,
                "http.user_agent": self.headers.get("User-Agent", ""),
            },
        )

        try:
            # Increment request counter
            client.increment_counter(
                "http.requests.total",
                attributes={"method": "GET", "path": self.path},
            )

            # Handle the request
            self.send_response(200)
            self.send_header("Content-type", "application/json")
            self.end_headers()
            self.wfile.write(b'{"message": "Hello from TelemetryFlow!"}')

            # Record success
            client.add_span_event(span_id, "response_sent", {"status_code": 200})

        except Exception as e:
            # Log error
            client.log_error(f"Request failed: {e}")
            client.increment_counter(
                "http.errors.total",
                attributes={"method": "GET", "path": self.path},
            )
            client.end_span(span_id, e)
            raise

        finally:
            # Record duration
            duration = time.time() - start_time
            client.record_histogram(
                "http.request.duration",
                duration,
                unit="s",
                attributes={"method": "GET", "path": self.path},
            )
            client.end_span(span_id)

    def log_message(self, format: str, *args: object) -> None:
        """Override to use TelemetryFlow logging."""
        client = get_client()
        client.log_info(format % args, {"handler": "http"})


def main() -> None:
    """Start the HTTP server with telemetry."""
    # Initialize TelemetryFlow SDK
    init()
    client = get_client()
    client.log_info("Starting HTTP server with TelemetryFlow instrumentation")

    # Start HTTP server
    server_address = ("", ${port})
    httpd = HTTPServer(server_address, InstrumentedHandler)

    print(f"Server running on http://localhost:${port}")
    print("Press Ctrl+C to stop")

    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        print("\nShutting down...")
    finally:
        httpd.shutdown()
        print("Server stopped.")


if __name__ == "__main__":
    main()
