"""Background worker with TelemetryFlow instrumentation example.

TelemetryFlow Python SDK - Community Enterprise Observability Platform (CEOP)
Copyright (c) 2024-2026 DevOpsCorner Indonesia. All rights reserved.

Auto-generated by telemetryflow-gen.
"""

import queue
import threading
import time
from dataclasses import dataclass
from typing import Any

from telemetry import init, get_client, SpanKind


@dataclass
class Job:
    """Job to be processed by worker."""

    id: str
    payload: dict[str, Any]


class Worker:
    """Background worker with telemetry instrumentation."""

    def __init__(self, num_workers: int = 5, queue_size: int = 100) -> None:
        self.job_queue: queue.Queue[Job | None] = queue.Queue(maxsize=queue_size)
        self.num_workers = num_workers
        self.workers: list[threading.Thread] = []
        self.running = False

    def start(self) -> None:
        """Start worker threads."""
        client = get_client()
        self.running = True

        for i in range(self.num_workers):
            worker = threading.Thread(target=self._worker_loop, args=(i,), daemon=True)
            worker.start()
            self.workers.append(worker)

        client.log_info(f"Started {self.num_workers} worker threads")
        client.record_gauge("worker.threads.active", self.num_workers)

    def stop(self) -> None:
        """Stop all worker threads."""
        client = get_client()
        self.running = False

        # Signal workers to stop
        for _ in self.workers:
            self.job_queue.put(None)

        # Wait for workers to finish
        for worker in self.workers:
            worker.join(timeout=5)

        client.log_info("All worker threads stopped")
        client.record_gauge("worker.threads.active", 0)

    def submit(self, job: Job) -> None:
        """Submit a job for processing."""
        client = get_client()
        self.job_queue.put(job)
        client.increment_counter("worker.jobs.submitted")
        client.record_gauge("worker.queue.size", self.job_queue.qsize())

    def _worker_loop(self, worker_id: int) -> None:
        """Worker thread main loop."""
        client = get_client()

        while self.running:
            try:
                job = self.job_queue.get(timeout=1)
                if job is None:
                    break

                self._process_job(worker_id, job)
                self.job_queue.task_done()

            except queue.Empty:
                continue
            except Exception as e:
                client.log_error(f"Worker {worker_id} error: {e}")
                client.increment_counter(
                    "worker.errors.total",
                    attributes={"worker_id": str(worker_id)},
                )

    def _process_job(self, worker_id: int, job: Job) -> None:
        """Process a single job with telemetry."""
        client = get_client()
        start_time = time.time()

        with client.span("process_job", SpanKind.CONSUMER, {"job.id": job.id}) as span_id:
            try:
                client.increment_counter(
                    "worker.jobs.processing",
                    attributes={"worker_id": str(worker_id)},
                )

                # Simulate work
                time.sleep(0.1)
                client.add_span_event(span_id, "job_validated")

                # More simulated work
                time.sleep(0.05)
                client.add_span_event(span_id, "job_processed")

                client.increment_counter("worker.jobs.completed")
                client.log_info(f"Job {job.id} completed", {"worker_id": worker_id})

            except Exception as e:
                client.increment_counter("worker.jobs.failed")
                client.log_error(f"Job {job.id} failed: {e}")
                raise

            finally:
                duration = time.time() - start_time
                client.record_histogram(
                    "worker.job.duration",
                    duration,
                    unit="s",
                    attributes={"worker_id": str(worker_id)},
                )


def main() -> None:
    """Run the worker example."""
    # Initialize TelemetryFlow SDK
    init()
    client = get_client()
    client.log_info("Starting worker example")

    # Create and start worker
    worker = Worker(num_workers=5, queue_size=100)
    worker.start()

    # Submit some jobs
    for i in range(20):
        job = Job(id=f"job-{i}", payload={"data": f"value-{i}"})
        worker.submit(job)

    # Wait for jobs to complete
    print("Submitted 20 jobs. Waiting for completion...")
    time.sleep(5)

    # Stop workers
    worker.stop()
    print("Worker example completed.")


if __name__ == "__main__":
    main()
