"""TelemetryFlow tracing helpers.

TelemetryFlow Python SDK - Community Enterprise Observability Platform (CEOP)
Copyright (c) 2024-2026 DevOpsCorner Indonesia. All rights reserved.

Auto-generated by telemetryflow-gen.
"""

from __future__ import annotations

from contextlib import contextmanager
from typing import TYPE_CHECKING, Generator

if TYPE_CHECKING:
    from typing import Any

from telemetryflow.application.commands import SpanKind

from . import get_client


@contextmanager
def span(
    name: str,
    kind: SpanKind = SpanKind.INTERNAL,
    attributes: dict[str, Any] | None = None,
) -> Generator[str, None, None]:
    """Create a trace span context manager.

    Args:
        name: Span name.
        kind: Span kind (SERVER, CLIENT, INTERNAL, PRODUCER, CONSUMER).
        attributes: Optional span attributes.

    Yields:
        str: Span ID for adding events.

    Example:
        >>> from telemetry.traces import span, SpanKind
        >>> with span("process_order", SpanKind.SERVER) as span_id:
        ...     # Do work
        ...     add_event(span_id, "order_validated")
    """
    client = get_client()
    with client.span(name, kind, attributes) as span_id:
        yield span_id


def start_span(
    name: str,
    kind: SpanKind = SpanKind.INTERNAL,
    attributes: dict[str, Any] | None = None,
) -> str:
    """Start a new trace span.

    Args:
        name: Span name.
        kind: Span kind.
        attributes: Optional span attributes.

    Returns:
        str: Span ID.

    Note:
        Remember to call end_span() when done.
    """
    client = get_client()
    return client.start_span(name, kind, attributes)


def end_span(span_id: str, error: Exception | None = None) -> None:
    """End a trace span.

    Args:
        span_id: Span ID from start_span().
        error: Optional exception if span ended with error.
    """
    client = get_client()
    client.end_span(span_id, error)


def add_event(
    span_id: str,
    name: str,
    attributes: dict[str, Any] | None = None,
) -> None:
    """Add an event to a span.

    Args:
        span_id: Span ID.
        name: Event name.
        attributes: Optional event attributes.

    Example:
        >>> add_event(span_id, "cache_hit", {"key": "user:123"})
    """
    client = get_client()
    client.add_span_event(span_id, name, attributes)


__all__ = [
    "span",
    "start_span",
    "end_span",
    "add_event",
    "SpanKind",
]
