"""Middleware Setup.

TelemetryFlow Python SDK - Community Enterprise Observability Platform (CEOP)
Copyright (c) 2024-2026 DevOpsCorner Indonesia. All rights reserved.

Auto-generated by telemetryflow-restapi.
"""

from __future__ import annotations

import time
from typing import TYPE_CHECKING

from flask import Flask, g, request

if TYPE_CHECKING:
    from ${module_name}.infrastructure.config import Config


def setup_middleware(app: Flask, config: "Config") -> None:
    """Setup application middleware.

    Args:
        app: Flask application.
        config: Application configuration.
    """
    # CORS
    if config.cors_enabled:
        from flask_cors import CORS
        CORS(app, resources={r"/api/*": {"origins": "*"}})

    # Rate limiting
    if config.rate_limit_enabled:
        from flask_limiter import Limiter
        from flask_limiter.util import get_remote_address
        Limiter(
            app=app,
            key_func=get_remote_address,
            default_limits=["100 per minute"],
        )

    # JWT
    from flask_jwt_extended import JWTManager
    JWTManager(app)

    # Swagger
    if config.swagger_enabled:
        from flasgger import Swagger
        Swagger(app, template={
            "info": {
                "title": config.service_name,
                "version": config.service_version,
                "description": f"{config.service_name} API Documentation",
            },
            "securityDefinitions": {
                "Bearer": {
                    "type": "apiKey",
                    "name": "Authorization",
                    "in": "header",
                    "description": "JWT Authorization header using the Bearer scheme. Example: \"Bearer {token}\""
                }
            }
        })

    # TelemetryFlow integration
    if config.telemetry_enabled:
        _setup_telemetry(app, config)

    # Request timing
    @app.before_request
    def before_request() -> None:
        g.start_time = time.time()

    @app.after_request
    def after_request(response):
        if hasattr(g, "start_time"):
            duration = time.time() - g.start_time
            response.headers["X-Response-Time"] = f"{duration:.3f}s"
        return response


def _setup_telemetry(app: Flask, config: "Config") -> None:
    """Setup TelemetryFlow integration.

    Args:
        app: Flask application.
        config: Application configuration.
    """
    try:
        from telemetryflow import TelemetryFlowBuilder
        from telemetryflow.middleware.flask import TelemetryFlowMiddleware

        client = (
            TelemetryFlowBuilder()
            .with_auto_configuration()
            .build()
        )
        client.initialize()

        # Add middleware
        TelemetryFlowMiddleware(app, client)

        app.config["TELEMETRY_CLIENT"] = client

        @app.teardown_appcontext
        def shutdown_telemetry(exception=None):
            pass  # Client shutdown handled by atexit

    except ImportError:
        app.logger.warning("TelemetryFlow SDK not installed. Telemetry disabled.")
    except Exception as e:
        app.logger.warning(f"Failed to initialize TelemetryFlow: {e}")
