# AUTOGENERATED! DO NOT EDIT! File to edit: ../0-utils.ipynb.

# %% auto 0
__all__ = ['iterable_have_common', 'check_state_tensor', 'check_quantum_gate', 'inverse_permutation', 'unify_tensor_dtypes', 'map_float_to_complex', 'view_gate_tensor_as_matrix', 'view_gate_matrix_as_tensor', 'load_iris', 'split_classification_dataset']

# %% ../0-utils.ipynb 0
from typing import List, Tuple
import torch

# %% ../0-utils.ipynb 2
def iterable_have_common(a: List[int] | Tuple[int], b: List[int] | Tuple[int]) -> bool:
    """
    Check if two iterables have any common elements.

    Args:
        a: First iterable.
        b: Second iterable.

    Returns:
        True if there are common elements, False otherwise.
    """
    assert isinstance(a, (list, tuple)), "First argument must be a list or tuple."
    assert isinstance(b, (list, tuple)), "Second argument must be a list or tuple."
    return len(set(a) & set(b)) > 0


def check_state_tensor(tensor: torch.Tensor):
    """
    Check if the tensor is a valid quantum state tensor.
    Args:
        tensor: The tensor to check.
    Raises:
        AssertionError: If the tensor is not a valid quantum state tensor.
    """
    assert isinstance(tensor, torch.Tensor), "quantum_state must be a torch.Tensor"
    assert tensor.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_state must be a float or complex tensor"
    )
    assert all(x == 2 for x in tensor.shape), (
        "quantum_state must be a tensor with all dimensions of size 2"
    )
    assert tensor.ndim > 0, "quantum_state must be a tensor with at least one dimension"


def check_quantum_gate(tensor: torch.Tensor, num_qubits: int | None = None) -> int:
    """
    Check if the tensor is a valid quantum gate tensor.
    Args:
        tensor: The tensor to check.
        num_qubits: The number of qubits.
    Returns:
        The number of qubits that the gate acts on.
    Raises:
        AssertionError: If the tensor is not a valid quantum gate tensor.
    """
    assert isinstance(tensor, torch.Tensor), "quantum_gate must be a torch.Tensor"
    assert tensor.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_gate must be a float or complex tensor"
    )
    assert tensor.ndim >= 2, "quantum_gate must be a tensor with at least two dimensions"
    assert tensor.ndim % 2 == 0, "quantum_gate must have an even number of dimensions"

    if tensor.ndim == 2:
        # in matrix form
        num_qubits = int(tensor.shape[0].bit_length() - 1) if num_qubits is None else num_qubits
        assert tensor.shape[0] == tensor.shape[1] == 2**num_qubits, (
            f"gate must be a square matrix with dimensions 2^num_qubits, got {tensor.shape}"
        )
    else:
        assert all(d == 2 for d in tensor.shape), "gate tensor must have all dimensions of size 2"
        num_qubits = tensor.ndim // 2 if num_qubits is None else num_qubits
        assert tensor.ndim == 2 * num_qubits, (
            f"gate tensor must have 2 * num_qubits dimensions, got {tensor.ndim}"
        )
    return num_qubits

# %% ../0-utils.ipynb 4
def inverse_permutation(permutation: List[int]) -> List[int]:
    permutation = torch.tensor(permutation, dtype=torch.long)
    inv = torch.empty_like(permutation)
    inv[permutation] = torch.arange(permutation.size(0))
    return inv.tolist()


def unify_tensor_dtypes(t1: torch.Tensor, t2: torch.Tensor) -> Tuple[torch.Tensor, torch.Tensor]:
    """
    Unify the dtypes of two tensors to the most appropriate type.
    Args:
        t1: First tensor.
        t2: Second tensor.
    Returns:
        Tuple of tensors with unified dtypes.
    Raises:
        AssertionError: if the dtypes of the tensors are not valid.
    """
    assert t1.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_state must be a float or complex tensor"
    )
    assert t2.dtype in [torch.float32, torch.float64, torch.complex64, torch.complex128], (
        "quantum_state must be a float or complex tensor"
    )
    if t1.dtype == t2.dtype:
        return t1, t2
    convert_dtypes = [
        (torch.float32, torch.complex64, torch.complex64),
        (torch.float64, torch.complex64, torch.complex128),
        (torch.float32, torch.complex128, torch.complex128),
        (torch.float64, torch.complex128, torch.complex128),
    ]
    for d1, d2, td in convert_dtypes:
        if (t1.dtype == d1 and t2.dtype == d2) or (t1.dtype == d2 and t2.dtype == d1):
            return t1.to(td), t2.to(td)
    raise_dtypes = [
        (torch.float32, torch.float64),
        (torch.complex64, torch.complex128),
    ]
    for d1, d2 in raise_dtypes:
        if (t1.dtype == d1 and t2.dtype == d2) or (t1.dtype == d2 and t2.dtype == d1):
            return t1.to(d2), t2.to(d2)

    raise Exception("Unreachable code in unify_tensor_dtypes")


def map_float_to_complex(
    *, tensor: torch.Tensor | None = None, dtype: torch.dtype | None = None
) -> torch.Tensor | torch.dtype:
    """
    Map a float tensor or dtype to a complex tensor or dtype.
    Args:
        tensor: The input tensor.
        dtype: The input dtype.
    Returns:
        The complex tensor or dtype.
    Raises:
        AssertionError: If neither tensor nor dtype is provided.
    """
    assert tensor is not None or dtype is not None, "Either tensor or dtype must be provided"
    original_dtype = tensor.dtype if tensor is not None else dtype
    assert original_dtype in [torch.float32, torch.float64], "dtype must be float32 or float64"
    to_dtype = torch.complex64 if original_dtype == torch.float32 else torch.complex128
    if tensor is not None:
        return tensor.to(to_dtype)
    return to_dtype


def view_gate_tensor_as_matrix(
    tensor: torch.Tensor, *, num_qubit: int | None = None
) -> torch.Tensor:
    """
    Convert a tensor representing a quantum gate into a matrix form.
    The tensor should have an even number of dimensions, each of size 2.

    Args:
        tensor (torch.Tensor): The tensor representing the quantum gate.
        num_qubit (int | None): The number of qubits the gate is acting on. If None, it is inferred from the tensor shape.
    Returns:
        torch.Tensor: The matrix form of the quantum gate tensor.
    """
    assert tensor.ndim % 2 == 0, "Tensor must have an even number of dimensions"
    assert all(d == 2 for d in tensor.shape), "Tensor dimensions must be 2"
    qubit_count = tensor.ndim // 2 if num_qubit is None else num_qubit
    check_quantum_gate(tensor, qubit_count)
    return tensor.view(2**qubit_count, 2**qubit_count)


def view_gate_matrix_as_tensor(
    tensor: torch.Tensor, *, num_qubit: int | None = None
) -> torch.Tensor:
    """
    Convert a matrix representing a quantum gate into a tensor form.
    The matrix should have dimensions (2^n, 2^n) for some n.

    Args:
        tensor (torch.Tensor): The matrix representing the quantum gate.
        num_qubit (int | None): The number of qubits the gate is acting on. If None, it is inferred from the matrix shape.
    Returns:
        torch.Tensor: The tensor form of the quantum gate matrix.
    """
    assert tensor.ndim == 2, "Matrix must have 2 dimensions"
    assert tensor.shape[0] == tensor.shape[1], "Matrix must be square"
    qubit_count = int(tensor.shape[0].bit_length() - 1) if num_qubit is None else num_qubit
    assert tensor.shape[0] == 2**qubit_count, (
        f"Matrix size must be (2^q, 2^q) for q qubits, but got {tensor.shape}"
    )
    check_quantum_gate(tensor, qubit_count)
    return tensor.view(*([2] * (qubit_count * 2)))

# %% ../0-utils.ipynb 7
def load_iris(*, force_single_precision=False) -> Tuple[torch.Tensor, torch.Tensor]:
    """
    Load the Iris dataset and return the data and targets.

    Args:
        force_single_precision: Whether to force the data to be single precision.
    Returns:
        Tuple[torch.Tensor, torch.Tensor]: The data and targets.
    """
    from sklearn import datasets

    iris = datasets.load_iris()
    data = torch.from_numpy(iris["data"])
    if force_single_precision:
        data = data.float()

    targets = torch.from_numpy(iris["target"]).to(torch.long)
    return data, targets

# %% ../3-5.ipynb 20
def split_classification_dataset(
    data: torch.Tensor, targets: torch.Tensor, ratio: float, shuffle: bool = True
) -> Tuple[torch.Tensor, torch.Tensor, torch.Tensor, torch.Tensor]:
    """
    Split the dataset into training and testing sets.

    Args:
        data (torch.Tensor): The data tensor.
        target (torch.Tensor): The target tensor.
        ratio (float): The ratio of the training set.
        shuffle (bool): Whether to shuffle the data.

    Returns:
        Tuple[torch.Tensor, torch.Tensor, torch.Tensor, torch.Tensor]: The training samples, training labels, testing samples, testing labels.
    """
    assert targets.dtype in [torch.int, torch.long], "target must be an integer tensor"
    assert 1.0 > ratio > 0.0, "ratio must be between 0 and 1"
    num_classes = torch.max(targets) + 1
    num_samples = data.shape[0]
    assert num_samples == targets.shape[0], "data and target must have the same number of samples"
    train_samples = []
    train_labels = []
    test_samples = []
    test_labels = []
    for class_idx in range(num_classes):
        sample_class_i = data[targets == class_idx]
        num_samples_class_i = sample_class_i.shape[0]
        num_train = int(num_samples_class_i * (1 - ratio))
        if shuffle:
            shuffle_idx = torch.randperm(num_samples_class_i)
            shuffled_sample_class_i = sample_class_i[shuffle_idx]
            train_sample = shuffled_sample_class_i[:num_train]
            train_samples.append(train_sample)
            test_sample = shuffled_sample_class_i[num_train:]
            test_samples.append(test_sample)

            train_labels.append(
                torch.full((train_sample.shape[0],), class_idx, dtype=targets.dtype)
            )
            test_labels.append(torch.full((test_sample.shape[0],), class_idx, dtype=targets.dtype))
        else:
            train_sample = sample_class_i[:num_train]
            train_samples.append(train_sample)
            test_sample = sample_class_i[num_train:]
            test_samples.append(test_sample)
            train_labels.append(
                torch.full((train_sample.shape[0],), class_idx, dtype=targets.dtype)
            )
            test_labels.append(torch.full((test_sample.shape[0],), class_idx, dtype=targets.dtype))

    train_samples = torch.cat(train_samples, dim=0)
    train_labels = torch.cat(train_labels, dim=0)
    test_samples = torch.cat(test_samples, dim=0)
    test_labels = torch.cat(test_labels, dim=0)
    return train_samples, train_labels, test_samples, test_labels