# AUTOGENERATED! DO NOT EDIT! File to edit: ../1-6.ipynb.

# %% auto 0
__all__ = ['rand_hermitian_matrix', 'rand_real_symmetric_matrix', 'eigs_power', 'eigs_power_ref']

# %% ../1-6.ipynb 0
import torch
from tqdm.auto import tqdm
import matplotlib.pyplot as plt

# %% ../1-6.ipynb 2
def rand_hermitian_matrix(dim: int) -> torch.Tensor:
    """
    Generate a random Hermitian matrix of given dimension.

    Args:
        dim (int): Dimension of the matrix.
    Returns:
        torch.Tensor: A Hermitian matrix of the specified dimension.
    """
    H = torch.randn(dim, dim, dtype=torch.complex64)
    H = H + H.conj().t()
    return H


# |export
def rand_real_symmetric_matrix(dim: int) -> torch.Tensor:
    """
    Generate a random real symmetric matrix of given dimension.

    Args:
        dim (int): Dimension of the matrix.
    Returns:
        torch.Tensor: A real symmetric matrix of the specified dimension.
    """
    mat = torch.randn(dim, dim, dtype=torch.float32)
    mat = (mat + mat.t()) / 2
    return mat

# %% ../1-6.ipynb 14
def eigs_power(mat: torch.Tensor, which: str, v0=None) -> (torch.Tensor, torch.Tensor):
    """
    ‘LM’ : Largest (in magnitude) eigenvalues.
    ‘SM’ : Smallest (in magnitude) eigenvalues.
    ‘LA’ : Largest (algebraic) eigenvalues.
    ‘SA’ : Smallest (algebraic) eigenvalues.

    Args:
        mat (torch.Tensor): Input matrix (real symmetric).
        which (str): Which eigenvalue to compute ('la', 'sa', 'lm', 'sm').
        v0 (torch.Tensor, optional): Initial vector. If None, a random vector is generated.
    Returns:
        (torch.Tensor, torch.Tensor): The computed eigenvalue and corresponding eigenvector.
    """
    which = which.lower()
    H = mat
    assert which in ["la", "sa", "lm", "sm"]
    assert H.allclose(H.t())
    assert len(H.shape) == 2

    TAU = 0.01
    if which == "la":
        rho = torch.matrix_exp(TAU * H)
    elif which == "lm":
        rho = torch.matrix_exp(TAU * torch.matrix_power(H, 2))
    elif which == "sa":
        rho = torch.matrix_exp(-TAU * H)
    elif which == "sm":
        rho = torch.matrix_exp(-TAU * torch.matrix_power(H, 2))
    else:
        raise NotImplementedError()

    ITER_NUM = 2000
    TOLERANCE = 1e-14
    if v0 is None:
        v = torch.randn(H.shape[1], dtype=H.dtype)
        v = v / v.norm()
    else:
        v = v0

    norm = 1.0
    for _ in tqdm(range(ITER_NUM)):
        v_next = rho @ v
        norm = v_next.norm()
        v_next /= norm
        diff = (v_next - v).norm()
        if diff < TOLERANCE:
            break
        v = v_next

    scaled_eigenvector = H.matmul(v)  # eigenvalue * eigenvector
    # correct sign due to squaring
    sign = torch.sign(v.dot(scaled_eigenvector))

    # TODO: Why we use `norm` instead of `scaled_eigenvector.norm()`?
    if which == "la":
        return torch.log(norm) / TAU, scaled_eigenvector / scaled_eigenvector.norm()
    elif which == "sa":
        return -torch.log(norm) / TAU, scaled_eigenvector / scaled_eigenvector.norm()
    elif which == "lm":
        return sign * torch.sqrt(
            torch.log(norm) / TAU
        ), scaled_eigenvector / scaled_eigenvector.norm()
    elif which == "sm":
        return sign * torch.sqrt(
            -torch.log(norm) / TAU
        ), scaled_eigenvector / scaled_eigenvector.norm()


# |export
def eigs_power_ref(mat, v0=None, which="la", tau=0.01, it_time=2000, tol=1e-14):
    """
    From https://github.com/ranshiju/Python-for-Tensor-Network-Tutorial/blob/4c89b0766159d3495122ec39339e7bd019f10fdf/Library/ExampleFun.py#L4

    :param mat: 输入矩阵（实对称阵）
    :param v0: 初始化向量，默认值为随机向量
    :param which: 计算哪个本征值与本征向量，
                  'la'为代数最大，'sa'为代数最小，'lm'为模最大，'sm'为模最小
    :param tau: 小的正实数，用以构造投影矩阵
    :param it_time: 最大迭代步数
    :param tol: 收敛阈值
    :return -tc.log(lm)/tau: 代数最大（tau>0）或最小（tau<0）的本征值
    :return v1: 对应的本征向量
    """
    tc = torch
    # 初始化向量
    if v0 is None:
        v0 = tc.randn(mat.shape[1], dtype=mat.dtype)
        v0 /= v0.norm()
    v1 = v0.clone()

    # 根据which给出投影矩阵
    tau = abs(tau)
    if which.lower() == "la":
        rho = tc.matrix_exp(tau * mat)
    elif which.lower() == "sa":
        rho = tc.matrix_exp(-tau * mat)
    elif which.lower() == "lm":
        rho = tc.matrix_exp(tau * (tc.matrix_power(mat, 2)))
    else:  # which.lower() == 'sm'
        rho = tc.matrix_exp(-tau * (tc.matrix_power(mat, 2)))

    lm = 1
    for n in range(it_time):  # 开始循环迭代
        v1 = rho.matmul(v0)  # 计算v1 = rho V0
        lm = v1.norm()  # 求本征值
        v1 /= lm  # 归一化v1
        # 判断收敛
        conv = (v1 - v0).norm()
        if conv < tol:
            break
        else:
            v0 = v1.clone()

    # 修正平方带来的符号丢失
    v1 = mat.matmul(v0)
    sign = tc.sign(v0.dot(v1))

    if which.lower() == "la":
        return tc.log(lm) / tau, v1 / v1.norm()
    elif which.lower() == "sa":
        return -tc.log(lm) / tau, v1 / v1.norm()
    elif which.lower() == "lm":
        return sign * tc.sqrt(tc.log(lm) / tau), v1 / v1.norm()
    else:  # which.lower() == 'sm'
        return sign * tc.sqrt(-tc.log(lm) / tau), v1 / v1.norm()
