# AUTOGENERATED! DO NOT EDIT! File to edit: ../3-5.ipynb.

# %% auto 0
__all__ = ['cossin_feature_map', 'feature_map_to_qubit_state']

# %% ../3-5.ipynb 15
import torch

# %% ../3-5.ipynb 16
def cossin_feature_map(samples: torch.Tensor, theta: float = 1.0) -> torch.Tensor:
    """
    Apply cossin feature mapping for qubit systems (d=2).

    See: https://proceedings.neurips.cc/paper/2016/hash/5314b9674c86e3f9d1ba25ef9bb32895-Abstract.html

    And also see https://www.perplexity.ai/search/do-you-know-the-cossin-feature-PidGvE9TTZSx9NaxAi23Gg

    Args:
        samples: Input tensor of shape (batch_size, feature_num)
        theta: Angle parameter (default: 1.0)

    Returns:
        Transformed features of shape (batch_size, feature_num, 2)
    """
    # Ensure samples is a 2D tensor
    if samples.ndimension() == 1:
        samples = samples.unsqueeze(0)  # Add batch dimension if missing

    # Calculate the cosine and sine components
    angle = samples * (theta * torch.pi)
    cos_features = torch.cos(angle)
    sin_features = torch.sin(angle)

    # Stack the features along a new dimension
    # Shape becomes (batch_size, feature_num, 2)
    stacked_features = torch.stack([cos_features, sin_features], dim=-1)

    return stacked_features

# %% ../3-6.ipynb 4
from einops import einsum


def feature_map_to_qubit_state(features: torch.Tensor) -> torch.Tensor:
    """
    Convert a feature tensor of shape (batch_size, feature_dim, 2) to a qubit state tensor of shape (batch_size, 2, ..., 2)
    """
    assert features.ndim == 3 and features.shape[2] == 2, (
        f"feature must be a 3D tensor of shape (batch_size, feature_dim, 2), but got {features.shape}"
    )
    feature_num = features.shape[1]
    features = torch.split(features, 1, dim=1)  # tuple of tensors of shape (batch_size, 1, 2)
    features = [
        feature.squeeze(1) for feature in features
    ]  # list of tensors of shape (batch_size, 2)
    einsum_expression = "{features} -> {tensor}".format(
        features=",".join([f"batch f{i}" for i in range(feature_num)]),
        tensor=" ".join(["batch"] + [f"f{i}" for i in range(feature_num)]),
    )
    qubit_states = einsum(*features, einsum_expression)  # (batch_size, 2, ..., 2)
    return qubit_states