"""长度为 5 的开放边界的 MPS"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../../4-1.ipynb.

# %% auto 0
__all__ = ['MPSType', 'gen_random_mps_tensors', 'calc_global_tensor_by_contract', 'calc_global_tensor_by_tensordot', 'calculate_mps_norm_factors', 'normalize_mps', 'calc_inner_product', 'orthogonalize_left2right_step', 'orthogonalize_right2left_step', 'orthogonalize_arange']

# %% ../../4-1.ipynb 2
import torch
from ..tensor_utils import tensor_contract
from typing import List
from enum import Enum

# %% ../../4-1.ipynb 3
class MPSType(Enum):
    """
    The type of the MPS
    """

    Open = "Open"
    Periodic = "Periodic"

    @staticmethod
    def get_mps_type(mps_tensors: List[torch.Tensor]) -> "MPSType":
        """
        Determine the type of the MPS
        """
        if mps_tensors[0].shape[0] == 1 and mps_tensors[-1].shape[2] == 1:
            return MPSType.Open
        else:
            return MPSType.Periodic


def gen_random_mps_tensors(
    length: int,
    physical_dim: int,
    virtual_dim: int,
    mps_type: MPSType,
    dtype: torch.dtype | None = None,
    device: torch.device | None = None,
) -> List[torch.Tensor]:
    """
    Generate random MPS tensors

    Args:
        length: int, the length of the MPS
        physical_dim: int, the physical dimension of the MPS
        virtual_dim: int, the virtual dimension of the MPS
        mps_type: MPSType, the type of the MPS
        dtype: torch.dtype, the dtype of the MPS
        device: torch.device, the device of the MPS

    Returns:
        List[torch.Tensor], the MPS tensors
    """
    if mps_type == MPSType.Open:
        mps_tensors = []
        mps_tensors.append(torch.randn(1, physical_dim, virtual_dim, dtype=dtype, device=device))
        rand_tensor = torch.randn(
            length - 2, virtual_dim, physical_dim, virtual_dim, dtype=dtype, device=device
        )
        for i in range(length - 2):
            mps_tensors.append(rand_tensor[i])
        mps_tensors.append(torch.randn(virtual_dim, physical_dim, 1, dtype=dtype, device=device))
        return mps_tensors
    elif mps_type == MPSType.Periodic:
        mps_tensor = torch.randn(
            length, virtual_dim, physical_dim, virtual_dim, dtype=dtype, device=device
        )
        return [mps_tensor[i] for i in range(length)]
    else:
        raise NotImplementedError(f"MPS type {mps_type} is not implemented")


def calc_global_tensor_by_contract(mps_tensors: List[torch.Tensor]) -> torch.Tensor:
    """
    Calculate the global tensor by contracting the MPS tensors

    Args:
        mps_tensors: List[torch.Tensor], the MPS tensors

    Returns:
        torch.Tensor, the global tensor
    """

    length = len(mps_tensors)
    dim_names = []
    for i in range(length):
        dim_name_list = [f"t{i}{j}" for j in range(3)]
        dim_names.append(dim_name_list)

    input_expression = ",".join([" ".join(dim_names[i]) for i in range(length)])

    contract_dims = []
    for i in range(length - 1):
        right_dim = dim_names[i][2]
        left_dim = dim_names[i + 1][0]
        contract_dims.append({right_dim, left_dim})

    mps_type = MPSType.get_mps_type(mps_tensors)

    if mps_type == MPSType.Periodic:
        front_left_dim = dim_names[0][0]
        end_right_dim = dim_names[-1][2]
        contract_dims.append({front_left_dim, end_right_dim})
        output_dims = [dim_name_list[1] for dim_name_list in dim_names]
    else:
        output_dims = (
            [dim_names[0][0]]
            + [dim_name_list[1] for dim_name_list in dim_names]
            + [dim_names[-1][-1]]
        )

    output_expression = " ".join(output_dims)

    return tensor_contract(
        *mps_tensors, ein_expr=f"{input_expression} -> {output_expression}", dims=contract_dims
    ).squeeze()


def calc_global_tensor_by_tensordot(mps_tensors: List[torch.Tensor]) -> torch.Tensor:
    """
    Calculate the global tensor by tensordot

    Args:
        mps_tensors: List[torch.Tensor], the MPS tensors

    Returns:
        torch.Tensor, the global tensor
    """
    length = len(mps_tensors)
    psi = mps_tensors[0]
    for i in range(1, length):
        psi = torch.tensordot(psi, mps_tensors[i], dims=([-1], [0]))

    mps_type = MPSType.get_mps_type(mps_tensors)

    if mps_type == MPSType.Periodic:
        return torch.einsum("a...a->...", psi)
    elif mps_type == MPSType.Open:
        return psi.squeeze()
    else:
        raise NotImplementedError(f"MPS type {mps_type} is not implemented")

# %% ../../4-1.ipynb 12
def calculate_mps_norm_factors(
    mps_tensors: List[torch.Tensor], __efficient_mode: bool = True
) -> torch.Tensor:
    """
    Calculate the norm factors of the MPS

    Args:
        mps_tensors: List[torch.Tensor], the MPS tensors
        __efficient_mode: bool, whether to use efficient mode, only for Open MPS

    Returns:
        torch.Tensor, the norm factors
    """

    assert len(mps_tensors) >= 1, "MPS must have at least one tensor"
    conjugates = [t.conj() for t in mps_tensors]
    length = len(conjugates)
    device = conjugates[0].device
    dtype = conjugates[0].dtype
    mps_type = MPSType.get_mps_type(mps_tensors)

    if mps_type == MPSType.Open:
        v = torch.ones(1, 1, dtype=dtype, device=device)  # dims: a b
        norm_factors = torch.empty(length, dtype=dtype, device=device)
        if __efficient_mode:
            for i in range(length):
                v = torch.einsum("ab,aix->bix", v, conjugates[i])
                v = torch.einsum("bix,biy->xy", v, mps_tensors[i])
                norm_factor = v.norm()
                v /= norm_factor
                norm_factors[i] = norm_factor
        else:
            for i in range(length):
                v = torch.einsum("ab,aix,biy->xy", v, conjugates[i], mps_tensors[i])
                norm_factor = v.norm()
                v /= norm_factor
                norm_factors[i] = norm_factor
        return norm_factors
    elif mps_type == MPSType.Periodic:
        virtual_dim = mps_tensors[0].shape[0]
        norm_factors = torch.empty(length, dtype=dtype, device=device)
        v = torch.eye(virtual_dim**2, dtype=dtype, device=device).reshape(
            virtual_dim, virtual_dim, virtual_dim, virtual_dim
        )
        for i in range(length):
            v = torch.einsum("uvap,adb,pdq->uvbq", v, conjugates[i], mps_tensors[i])
            norm_factor = v.norm()
            v /= norm_factor
            norm_factors[i] = norm_factor

        final_norm_factor = torch.einsum("acac->", v)
        norm_factors[-1] *= final_norm_factor
        return norm_factors
    else:
        raise NotImplementedError(f"MPS type {mps_type} is not implemented")

# %% ../../4-1.ipynb 18
def normalize_mps(mps_tensors: List[torch.Tensor]) -> List[torch.Tensor]:
    """
    Normalize the MPS

    Args:
        mps_tensors: List[torch.Tensor], the MPS tensors

    Returns:
        List[torch.Tensor], the normalized MPS tensors
    """
    length = len(mps_tensors)
    norms = calculate_mps_norm_factors(mps_tensors)
    normalization_factors = 1 / norms.sqrt()
    middle_tensors = torch.stack(
        mps_tensors[1:-1]
    )  # (length - 2, virtual_dim, physical_dim, virtual_dim)
    middle_factors = normalization_factors[1:-1].reshape(-1, 1, 1, 1)
    normalized_middle_tensors = middle_tensors * middle_factors
    front_tensor = mps_tensors[0] * normalization_factors[0]
    end_tensor = mps_tensors[-1] * normalization_factors[-1]
    return [front_tensor] + [normalized_middle_tensors[i] for i in range(length - 2)] + [end_tensor]

# %% ../../4-1.ipynb 23
def calc_inner_product(mps0: List[torch.Tensor], mps1: List[torch.Tensor]) -> torch.Tensor:
    """
    Calculate the inner product of two MPS of length N

    Args:
        mps0: List[torch.Tensor], the first MPS that to be conjugated
        mps1: List[torch.Tensor], the second MPS

    Returns:
        torch.Tensor, inner product factors of lengh N + 1
    """
    assert len(mps0) == len(mps1)
    assert mps0[0].shape[0] == mps0[-1].shape[-1]
    assert mps1[0].shape[0] == mps1[-1].shape[-1]
    assert mps0[0].dtype == mps1[0].dtype
    assert mps0[0].device == mps1[0].device
    endpoint_virtual_dim1 = mps0[0].shape[0]
    endpoint_virtual_dim2 = mps1[0].shape[0]
    length = len(mps0)
    dtype = mps0[0].dtype
    device = mps0[0].device
    v0 = torch.eye(endpoint_virtual_dim1, dtype=dtype, device=device)
    v1 = torch.eye(endpoint_virtual_dim2, dtype=dtype, device=device)
    # cross product of two identity matrices
    v = torch.einsum("ab,xy->axby", v0, v1)

    inner_product_factors = torch.empty(length + 1, dtype=dtype, device=device)
    for i in range(length):
        # a: mps0_conj left
        # b: mps0_conj right
        # d: physical dimension
        # p: mps1 left
        # q: mps1 right
        # u: mps0 virtual dim
        # v: mps1 virtual dim
        v = torch.einsum("uvap,adb,pdq->uvbq", v, mps0[i].conj(), mps1[i])
        product_factor = v.norm()
        v /= product_factor
        inner_product_factors[i] = product_factor

    if v.numel() > 1:
        # meaning if any of two MPS is periodic
        product_factor = torch.einsum("acac->", v)
        inner_product_factors[-1] = product_factor
    else:
        # both are open MPS
        inner_product_factors[-1] = v[0, 0, 0, 0]

    return inner_product_factors

# %% ../../4-2.ipynb 4
from typing import Literal, Tuple


def orthogonalize_left2right_step(
    mps_tensors: List[torch.Tensor],
    local_tensor_idx: int,
    mode: Literal["svd", "qr"],
    truncate_dim: int | None = None,
    return_locals: bool = False,
) -> List[torch.Tensor] | Tuple[torch.Tensor, torch.Tensor]:
    """
    One step of orthogonalization from left to right, which will make the local tensor isometric and the right one to it transformed.

    Args:
        mps_tensors: List[torch.Tensor], MPS tensors
        local_tensor_idx: int, the index of the local tensor to be orthogonalized.
        mode: Literal["svd", "qr"], the mode of orthogonalization.
        truncate_dim: int | None, the dimension to be truncated. If None, no truncation will be performed.
        return_locals: bool, whether to return the local tensors. If True, only the local and the right one will be returned.

    Returns:
        List[torch.Tensor], the list of tensors after one step of orthogonalization from left to right.
    """
    length = len(mps_tensors)
    assert length > 1, "mps_tensors must have at least 2 tensors"
    assert 0 <= local_tensor_idx < length - 1, "local_tensor_idx must be in [0, length - 2]"
    mode = mode.lower()
    assert mode in ["svd", "qr"], "mode must be either 'svd' or 'qr'"
    local_tensor = mps_tensors[local_tensor_idx]
    shape = local_tensor.shape  # (virtual_dim, physical_dim, virtual_dim)
    if truncate_dim is not None:
        virtual_dim = shape[2]
        assert virtual_dim > truncate_dim > 0, (
            "truncate_dim must be positive and less than virtual_dim"
        )
        assert mode == "svd", "mode must be 'svd' when truncate_dim is provided"
        need_truncate = True
    else:
        need_truncate = False

    view_matrix = local_tensor.view(-1, shape[2])

    if mode == "svd":
        u, lm, v = torch.linalg.svd(view_matrix, full_matrices=False)
        if need_truncate:
            u = u[:, :truncate_dim]
            lm = lm[:truncate_dim]  # (truncate_dim)
            v = v[:truncate_dim, :]  # (truncate_dim, virtual_dim)
            r = lm.unsqueeze(1) * v  # (truncate_dim, virtual_dim)
        else:
            r = lm.unsqueeze(1) * v  # (virtual_dim, virtual_dim)
    else:
        u, r = torch.linalg.qr(view_matrix)

    new_local_tensor = u.reshape(
        shape[0], shape[1], -1
    )  # (virtual_dim, physical_dim, virtual_dim or truncate_dim)
    local_tensor_right = mps_tensors[
        local_tensor_idx + 1
    ]  # (virtual_dim, physical_dim, virtual_dim)
    new_local_tensor_right = torch.einsum("ab,bcd->acd", r, local_tensor_right)
    if return_locals:
        return new_local_tensor, new_local_tensor_right
    else:
        return (
            mps_tensors[:local_tensor_idx]
            + [new_local_tensor, new_local_tensor_right]
            + mps_tensors[local_tensor_idx + 2 :]
        )


def orthogonalize_right2left_step(
    mps_tensors: List[torch.Tensor],
    local_tensor_idx: int,
    mode: Literal["svd", "qr"],
    truncate_dim: int | None = None,
    return_locals: bool = False,
) -> List[torch.Tensor] | Tuple[torch.Tensor, torch.Tensor]:
    """
    One step of orthogonalization from right to left, which will make the local tensor isometric and the left one to it transformed.

    Args:
        mps_tensors: List[torch.Tensor], MPS tensors
        local_tensor_idx: int, the index of the local tensor to be orthogonalized.
        mode: Literal["svd", "qr"], the mode of orthogonalization.
        truncate_dim: int | None, the dimension to be truncated. If None, no truncation will be performed.
        return_locals: bool, whether to return the local tensors. If True, only the local and the left one will be returned.

    Returns:
        List[torch.Tensor], the list of tensors after one step of orthogonalization from right to left.
    """
    length = len(mps_tensors)
    assert length > 1, "mps_tensors must have at least 2 tensors"
    assert 1 <= local_tensor_idx < length, "local_tensor_idx must be in [1, length - 1]"
    mode = mode.lower()
    assert mode in ["svd", "qr"], "mode must be either 'svd' or 'qr'"
    local_tensor = mps_tensors[local_tensor_idx]
    shape = local_tensor.shape  # (virtual_dim, physical_dim, virtual_dim)
    if truncate_dim is not None:
        virtual_dim = shape[0]
        assert virtual_dim > truncate_dim > 0, (
            "truncate_dim must be positive and less than virtual_dim"
        )
        assert mode == "svd", "mode must be 'svd' when truncate_dim is provided"
        need_truncate = True
    else:
        need_truncate = False

    view_matrix = local_tensor.view(
        shape[0], -1
    ).t()  # (virtual_dim, virtual_dim * physical_dim) -> (virtual_dim * physical_dim, virtual_dim)
    if mode == "svd":
        u, lm, v = torch.linalg.svd(view_matrix, full_matrices=False)
        if need_truncate:
            u = u[:, :truncate_dim]
            lm = lm[:truncate_dim]  # (truncate_dim)
            v = v[:truncate_dim, :]  # (truncate_dim, virtual_dim)
            r = lm.unsqueeze(1) * v  # (truncate_dim, virtual_dim)
        else:
            r = lm.unsqueeze(1) * v  # (virtual_dim, virtual_dim)
    else:
        u, r = torch.linalg.qr(view_matrix)

    new_local_tensor = u.t().reshape(
        -1, shape[1], shape[2]
    )  # (virtual_dim or truncate_dim, physical_dim, virtual_dim)
    local_tensor_left = mps_tensors[
        local_tensor_idx - 1
    ]  # (virtual_dim, physical_dim, virtual_dim)
    new_local_tensor_left = torch.einsum("abc,dc->abd", local_tensor_left, r)
    if return_locals:
        return new_local_tensor_left, new_local_tensor
    else:
        return (
            mps_tensors[: local_tensor_idx - 1]
            + [new_local_tensor_left, new_local_tensor]
            + mps_tensors[local_tensor_idx + 1 :]
        )

# %% ../../4-2.ipynb 8
def orthogonalize_arange(
    mps_tensors: List[torch.Tensor],
    start_idx: int,
    end_idx: int,
    mode: Literal["svd", "qr"],
    truncate_dim: int | None = None,
    return_changed: bool = False,
) -> List[torch.Tensor] | Tuple[List[torch.Tensor], List[int]]:
    """
    Perform orthogonalization on the range of tensors.

    Args:
        mps_tensors: List[torch.Tensor], MPS tensors
        start_idx: int, the start index of the range
        end_idx: int, the end index of the range
        mode: Literal["svd", "qr"], the mode of orthogonalization
        truncate_dim: int | None, the dimension to be truncated. If None, no truncation will be performed.
        return_changed: bool, whether to return the changed tensors. If True, changed tensors' indices will be returned as well.

    Returns:
        List[torch.Tensor], the list of tensors after orthogonalization
    """
    length = len(mps_tensors)
    assert length > 1, "mps_tensors must have at least 2 tensors"
    assert 0 <= start_idx < length and 0 <= end_idx < length, (
        "start_idx and end_idx must be in [0, length - 1]"
    )
    mps_tensors = [m for m in mps_tensors]
    changed_indices = set()
    if start_idx < end_idx:
        for idx in range(start_idx, end_idx, 1):
            local, local_right = orthogonalize_left2right_step(
                mps_tensors, idx, mode, truncate_dim, return_locals=True
            )
            mps_tensors[idx] = local
            mps_tensors[idx + 1] = local_right
            changed_indices.add(idx)
            changed_indices.add(idx + 1)
    elif start_idx > end_idx:
        for idx in range(start_idx, end_idx, -1):
            local_left, local = orthogonalize_right2left_step(
                mps_tensors, idx, mode, truncate_dim, return_locals=True
            )
            mps_tensors[idx - 1] = local_left
            mps_tensors[idx] = local
            changed_indices.add(idx - 1)
            changed_indices.add(idx)
    else:
        # do nothing when start_idx == end_idx
        pass

    if return_changed:
        changed_indices = list(changed_indices)
        changed_indices.sort()
        return mps_tensors, changed_indices
    else:
        return mps_tensors